<?php
namespace App\Model\Table;

use App\Model\Entity\Diagnostic;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;

/**
 * Diagnostics Model
 *
 * @property \Cake\ORM\Association\BelongsTo $Patients
 * @property \Cake\ORM\Association\BelongsTo $Hospitalizations
 * @property \Cake\ORM\Association\BelongsTo $Origins
 * @property \Cake\ORM\Association\BelongsTo $MedicalCares
 * @property \Cake\ORM\Association\BelongsToMany $Comorbidities
 * @property \Cake\ORM\Association\BelongsToMany $OtherComorbidities
 */
class DiagnosticsTable extends Table
{

    /**
     * Initialize method
     *
     * @param array $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config)
    {
        parent::initialize($config);

        $this->table('diagnostics');
        $this->displayField('id');
        $this->primaryKey('id');

        $this->addBehavior('Timestamp');

        $this->belongsTo('Patients', [
            'foreignKey' => 'patient_id',
            'joinType'   => 'INNER',
        ]);
        $this->belongsTo('Hospitalizations', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->belongsTo('Origins', [
            'foreignKey' => 'origin_id',
        ]);
        $this->belongsTo('MedicalCares', [
            'foreignKey' => 'medical_care_id',
        ]);

        $this->belongsToMany('Comorbidities', [
            'foreignKey'       => 'diagnostic_id',
            'targetForeignKey' => 'comorbidity_id',
            'joinTable'        => 'comorbidities_diagnostics',
        ]);

        $this->belongsToMany('OtherComorbidities', [
            'foreignKey'       => 'diagnostic_id',
            'targetForeignKey' => 'other_comorbidity_id',
            'joinTable'        => 'other_comorbidities_diagnostics',
        ]);

        $this->belongsToMany('DefectsAndAnomalies', [
            'foreignKey'       => 'diagnostic_id',
            'targetForeignKey' => 'defects_and_anomaly_id',
            'joinTable'        => 'defects_and_anomalies_diagnostics',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator)
    {
        $validator
            ->integer('id')
            ->allowEmpty('id', 'create');

        $validator
            ->allowEmpty('origin');

        $validator
            ->requirePresence('type', 'create')
            ->notEmpty('type');

        $validator
            ->requirePresence('description', 'create')
            ->notEmpty('description');

        $validator
            ->boolean('is_possible_diagnosis')
            ->allowEmpty('is_possible_diagnosis');

        $validator
        //->date('diagnostic_date')
        ->requirePresence('diagnostic_date', 'create')
            ->notEmpty('diagnostic_date');

        $validator
            ->allowEmpty('emergency_surgery');

        $validator
            ->allowEmpty('routine_surgery');

        $validator
            ->allowEmpty('clinic');

        $validator
            ->allowEmpty('grave_comorbidities');

        $validator
            ->allowEmpty('other_comorbidities');

        return $validator;
    }

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules)
    {
        $rules->add($rules->existsIn(['patient_id'], 'Patients'));
        $rules->add($rules->existsIn(['hospitalization_id'], 'Hospitalizations'));
        $rules->add($rules->existsIn(['origin_id'], 'Origins'));
        $rules->add($rules->existsIn(['medical_care_id'], 'MedicalCares'));
        return $rules;
    }
}
