<?php
namespace App\Controller;

use App\Controller\AppController;

/**
 * ParticipatingUsers Controller
 *
 * @property \App\Model\Table\ParticipatingUsersTable $ParticipatingUsers
 */
class ParticipatingUsersController extends AppController
{
    public function initialize()
    {
        parent::initialize();
        $this->loadComponent('Auth', [
            'authorize' => 'Controller',
        ]);
        $this->loadComponent('Permission');

        // View or Controller
        //$this->set('title', 'Bundle Diário');

        $session = $this->request->session();

        $pacient_id = $session->read('selected-pacient');
        $this->set('pacient_id', $pacient_id);

        $unit_id = $session->read('selected-unit');
        $this->set('unit_id', $unit_id);

        $bundle_id = $session->read('bundle_id');
        $this->set('bundle_id', $bundle_id);

        $hospitalization_id = $session->read('selected-hospitalization');
        $this->set('hospitalization_id', $hospitalization_id);
    }

    public function isAuthorized($user)
    {
        $action = $this->request->params['action'];

        return $this->Permission->hasPermission($user['role_name'], 'PavPreventionBundles', $action);
    }

    /**
     * Index method
     *
     * @return \Cake\Network\Response|null
     */
    public function index($participant_type)
    {
        $new_btn_label = null;
        switch ($participant_type) {
            case 1:
                $new_btn_label = 'Nova Equipe';
                break;            
            case 2:
                $new_btn_label = 'Nova Gestão de Protocolo';
                break;
            case 3:
                $new_btn_label = 'Nova Comissão';
                break;
            default:
                $new_btn_label = 'Novo';
                break;
        }

        $this->set('title', $this->ParticipatingUsers->getTitle($participant_type));

        $this->paginate = [
            'contain' => ['Units'],
            'conditions' => ['ParticipatingUsers.type' => $participant_type],
        ];
        $participatingUsers = $this->paginate($this->ParticipatingUsers);

        $this->set(compact('participatingUsers'));
        $this->set('_serialize', ['participatingUsers']);
        $this->set('participant_type', $participant_type);
        $this->set('new_btn_label', $new_btn_label);
    }

    /**
     * View method
     *
     * @param string|null $id Participating User id.
     * @return \Cake\Network\Response|null
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $participatingUser = $this->ParticipatingUsers->get($id, [
            'contain' => ['Units', 'Professionals'],
        ]);

        $this->set('participatingUser', $participatingUser);
        $this->set('_serialize', ['participatingUser']);
    }

    /**
     * Add method
     *
     * @return \Cake\Network\Response|void Redirects on successful add, renders view otherwise.
     */
    public function add($participant_type = null)
    {
        $this->set('title', $this->ParticipatingUsers->getTitle($participant_type));

        $participatingUser = $this->ParticipatingUsers->newEntity();

        if ($this->request->is('post')) {
            $participatingUser = $this->ParticipatingUsers->patchEntity($participatingUser, $this->request->data);

            if ($this->ParticipatingUsers->save($participatingUser)) {
                $this->Flash->success(__('The participating user has been saved.'));
                return $this->redirect(['action' => 'index', $participant_type]);
            } else {
                $this->Flash->error(__('The participating user could not be saved. Please, try again.'));
            }
        }

        $units = $this->ParticipatingUsers->Units->find('list', ['limit' => 200]);
        $professionals = $this->ParticipatingUsers->Professionals->find('list', ['limit' => 200]);
        $this->set(compact('participatingUser', 'units', 'professionals'));
        $this->set('_serialize', ['participatingUser']);
        $this->set('participant_type', $participant_type);        
    }

    /**
     * Edit method
     *
     * @param string|null $id Participating User id.
     * @return \Cake\Network\Response|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Network\Exception\NotFoundException When record not found.
     */
    public function edit($participant_type = null, $id = null)
    {
        $this->set('title', $this->ParticipatingUsers->getTitle($participant_type)); 

        $participatingUser = $this->ParticipatingUsers->get($id, [
            'contain' => ['Professionals'],
        ]);

        if ($this->request->is(['patch', 'post', 'put'])) {
            $participatingUser = $this->ParticipatingUsers->patchEntity($participatingUser, $this->request->data);

            if ($this->ParticipatingUsers->save($participatingUser)) {
                $this->Flash->success(__('The participating user has been saved.'));
                return $this->redirect(['action' => 'index', $participant_type]);
            } else {
                $this->Flash->error(__('The participating user could not be saved. Please, try again.'));
            }
        }

        $units = $this->ParticipatingUsers->Units->find('list', ['limit' => 200]);
        $professionals = $this->ParticipatingUsers->Professionals->find('list', ['limit' => 200]);
        $this->set(compact('participatingUser', 'units', 'professionals'));
        $this->set('_serialize', ['participatingUser']);
        $this->set('participant_type', $participant_type);
    }

    /**
     * Delete method
     *
     * @param string|null $id Participating User id.
     * @return \Cake\Network\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $participatingUser = $this->ParticipatingUsers->get($id);

        if ($this->ParticipatingUsers->delete($participatingUser)) {
            $this->Flash->success(__('The participating user has been deleted.'));
        } else {
            $this->Flash->error(__('The participating user could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }
}
