<?php
namespace App\Model\Table;

use App\Model\Entity\Hospitalization;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\ORM\TableRegistry;
use Cake\Validation\Validator;
use MedidasDispersao\MedidasDispersao;

require_once ROOT . DS . "vendor" . DS . "Zurubabel" . DS . "MedidasDispersao.php";

/**
 * Hospitalizations Model
 *
 * @property \Cake\ORM\Association\BelongsTo $Patients
 * @property \Cake\ORM\Association\BelongsTo $Screenings
 * @property \Cake\ORM\Association\BelongsTo $Origins
 * @property \Cake\ORM\Association\BelongsTo $Units
 * @property \Cake\ORM\Association\BelongsTo $Sectors
 * @property \Cake\ORM\Association\BelongsTo $MedicalCares
 * @property \Cake\ORM\Association\HasMany $ApacheIiScores
 * @property \Cake\ORM\Association\HasMany $BarthelIndexes
 * @property \Cake\ORM\Association\HasMany $BradenScales
 * @property \Cake\ORM\Association\HasMany $Bundles
 * @property \Cake\ORM\Association\HasMany $CardiacSurgeryChecklists
 * @property \Cake\ORM\Association\HasMany $ChestPainProtocols
 * @property \Cake\ORM\Association\HasMany $ClinicTevProtocols
 * @property \Cake\ORM\Association\HasMany $CribScores
 * @property \Cake\ORM\Association\HasMany $Cultures
 * @property \Cake\ORM\Association\HasMany $CurbScores
 * @property \Cake\ORM\Association\HasMany $DailySafetyChecklists
 * @property \Cake\ORM\Association\HasMany $Diagnostics
 * @property \Cake\ORM\Association\HasMany $Discharges
 * @property \Cake\ORM\Association\HasMany $FugulinBradenScales
 * @property \Cake\ORM\Association\HasMany $HistoricPatientMovements
 * @property \Cake\ORM\Association\HasMany $IdentificationOfRisks
 * @property \Cake\ORM\Association\HasMany $Infections
 * @property \Cake\ORM\Association\HasMany $InformedConsentForms
 * @property \Cake\ORM\Association\HasMany $ItuPreventionBundles
 * @property \Cake\ORM\Association\HasMany $MedicalBulletins
 * @property \Cake\ORM\Association\HasMany $MedicalOpinionRequests
 * @property \Cake\ORM\Association\HasMany $NasScores
 * @property \Cake\ORM\Association\HasMany $NeonatalChecklists
 * @property \Cake\ORM\Association\HasMany $NutritionalAssessments
 * @property \Cake\ORM\Association\HasMany $ObstetricSurgeryChecklists
 * @property \Cake\ORM\Association\HasMany $PavPreventionBundles
 * @property \Cake\ORM\Association\HasMany $PreAnestheticVisits
 * @property \Cake\ORM\Association\HasMany $PreDeliricScores
 * @property \Cake\ORM\Association\HasMany $Prescriptions
 * @property \Cake\ORM\Association\HasMany $SafeSurgeryBundles
 * @property \Cake\ORM\Association\HasMany $SapsIiiScores
 * @property \Cake\ORM\Association\HasMany $SepseProtocols
 * @property \Cake\ORM\Association\HasMany $SkinIntegrities
 * @property \Cake\ORM\Association\HasMany $SnapIiScores
 * @property \Cake\ORM\Association\HasMany $SofaEscores
 * @property \Cake\ORM\Association\HasMany $StrengthAssessmentScales
 * @property \Cake\ORM\Association\HasMany $SupportiveMeasures
 * @property \Cake\ORM\Association\HasMany $Surgeries
 * @property \Cake\ORM\Association\HasMany $SurgicalChecklists
 * @property \Cake\ORM\Association\HasMany $SurgicalTevProtocols
 * @property \Cake\ORM\Association\HasMany $SuspectedMeProtocols
 * @property \Cake\ORM\Association\HasMany $VentilatoryCarePlans
 */
class HospitalizationsTable extends Table
{

    /**
     * Initialize method
     *
     * @param array $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config)
    {
        parent::initialize($config);

        $this->table('hospitalizations');
        $this->displayField('id');
        $this->primaryKey('id');

        $this->addBehavior('Timestamp');

        $this->belongsTo('Patients', [
            'foreignKey' => 'patient_id',
            'joinType' => 'INNER',
        ]);
        $this->belongsTo('Screenings', [
            'foreignKey' => 'screening_id',
        ]);
        $this->belongsTo('Origins', [
            'foreignKey' => 'origin_id',
            'joinType' => 'INNER',
        ]);
        $this->belongsTo('Professionals', [
            'foreignKey' => 'professional_id',
            'joinType' => 'LEFT',
        ]);
        $this->belongsTo('Units', [
            'foreignKey' => 'unit_id',
            'joinType' => 'INNER',
        ]);
        $this->belongsTo('Sectors', [
            'foreignKey' => 'sector_id',
        ]);
        $this->belongsTo('MedicalCares', [
            'foreignKey' => 'medical_care_id',
        ]);
        $this->hasMany('ApacheIiScores', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('BarthelIndexes', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('BradenScales', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('Bundles', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('CardiacSurgeryChecklists', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('ChestPainProtocols', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('ClinicTevProtocols', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('CribScores', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('Cultures', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('CurbScores', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('DailySafetyChecklists', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('Diagnostics', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('Discharges', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('FugulinBradenScales', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('HistoricPatientMovements', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('IdentificationOfRisks', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('Infections', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('InformedConsentForms', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('ItuPreventionBundles', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('MedicalBulletins', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('MedicalOpinionRequests', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('NasScores', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('NeonatalChecklists', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('NutritionalAssessments', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('ObstetricSurgeryChecklists', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('PavPreventionBundles', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('PreAnestheticVisits', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('SurgicalRisks', [
            'foreignKey' => 'hospitalization_id',
        ]);        
        $this->hasMany('PreDeliricScores', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('Prescriptions', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('SafeSurgeryBundles', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('SapsIiiScores', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('SepseProtocols', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('SkinIntegrities', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('SnapIiScores', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('SofaEscores', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('StrengthAssessmentScales', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('SupportiveMeasures', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('Surgeries', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('SurgicalChecklists', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('SurgicalTevProtocols', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('SuspectedMeProtocols', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->hasMany('VentilatoryCarePlans', [
            'foreignKey' => 'hospitalization_id',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator)
    {
        $validator
            ->integer('id')
            ->allowEmpty('id', 'create');

        $validator
            ->allowEmpty('health_care_number');

        $validator
            ->allowEmpty('type');

        $validator
            ->allowEmpty('complementary_health');

        $validator
        //->date('expected_discharge_day')
        ->allowEmpty('expected_discharge_day');

        $validator
            ->boolean('readmission_to_hospital_until_30_days')
            ->allowEmpty('readmission_to_hospital_until_30_days');

        $validator
            ->boolean('rehospitalization_up_24')
            ->allowEmpty('rehospitalization_up_24');

        $validator
            ->numeric('weight')
            ->allowEmpty('weight');

        $validator
            ->numeric('height')
            ->allowEmpty('height');

        $validator
            ->allowEmpty('systemic_diagnosis');

        $validator
            ->allowEmpty('numero_do_prontuario');

        $validator
            ->numeric('imc')
            ->allowEmpty('imc');

        return $validator;
    }

    public function getRelatorioGeralData($hospitalizations, $startDate, $endDate, $sectors, $diagnostics)
    {
        $lastDayOfMonth = new \DateTime($endDate->format("Y/m/d"));
        $lastDayOfMonth->modify('last day of this month')->setTime(23, 59, 59);
        $lastDayOfMonth = (integer) $lastDayOfMonth->format("d");
        $origin_table = TableRegistry::get('Origins');
        $health_insurance_plans_table = TableRegistry::get('HealthInsurancePlans');

        $origins = $origin_table->find();

        $health_insurance_plans = $health_insurance_plans_table->find();

        $total_dias_no_periodo = $startDate->diff($endDate)->days;

        $calc_obj = new MedidasDispersao();

        $data = [
            'tot_pacientes' => 0,

            'tot_internacoes_no_periodo' => 0,

            'tot_pacientes_com_alta' => 0,
            'percen_pacientes_com_alta' => 0,

            'tot_pacientes_com_obito' => 0,
            'percen_pacientes_com_obito' => 0,

            'tempo_medio_permanencia' => 0,

            'sectors' => [],
            'principais_destinos' => [],

        ];

        $diagnostics_formatted = [];
        $systemic_diagnosis_formatted = [];

        foreach ($this->getSystemicDiagnosis() as $systemic_dianosis)
        {
            $systemic_dianosis_arr = [
                'name' => $systemic_dianosis,
                'tot_pacientes' => 0,
                'percen_pacientes' => 0,
                'arr_pacientes' => 0,
            ];

            array_push($systemic_diagnosis_formatted, $systemic_dianosis_arr);
        }

// foreach ($diagnostics as $diagnostic)

// {

//     $diagnostic_arr = [

//         'name' => $diagnostic['description'],

//         'tot' => 0,

//         'percen' => 0,

//         'tot_secundario' => 0,

//         'percen_secundario' => 0,

//         'tot_comorbidade' => 0,

//         'percen_comorbidade' => 0,

//     ];

//     //array_push($diagnostics_formatted, $diagnostic_arr);
        // }

        $health_insurance_plans_formatted = [];

        foreach ($health_insurance_plans as $key => $health_insurance_plan)
        {
            $health_insurance_plan_id = $health_insurance_plan['id'];

            $health_insurance_plans_formatted[$health_insurance_plan_id] = [
                'name' => $health_insurance_plan['name'],
                'tot' => 0,
                'percen' => 0,
            ];
            //array_push($health_insurance_plans_formatted, $health_insurance_plan_arr);
        }

        $diagnostics_t = $this->returnDiagnostics();

        foreach ($sectors as $sector)
        {
            $sector_id = $sector['id'];
            $tot_de_leitos = $sector['limit_of_rooms'];

            $data['sectors'][$sector_id] = [
                'name' => $sector['name'],

                'arr_pacientes_atendidos' => [],
                'tot_pacientes_atendidos' => 0,

                'arr_tot_internacoes' => [],
                'tot_internacoes_no_periodo' => 0,

                'arr_tot_obito' => [],
                'tot_obito' => 0,
                'arr_idade_obito' => [],
                'media_idade_obito' => 0,

                'arr_idade_alta' => [],
                'media_idade_alta' => 0,

                'arr_origem' => [],
                'percen_obito' => 0,

                'arr_tot_alta' => [],
                'tot_alta' => 0,
                'percen_alta' => 0,

                'arr_tempo_medio_permanencia' => [],
                'tempo_medio_permanencia' => 0,

                'tempo_minimo_permanencia' => 0,
                'tempo_maximo_permanencia' => 0,

                'arr_tempo_medio_permanencia_obito' => [],
                'tempo_medio_permanencia_obito' => 0,

                'arr_tempo_medio_permanencia_alta' => [],
                'tempo_medio_permanencia_alta' => 0,

                'arr_tempo_medio_duracao_alta_e_alta_efetiva' => [],
                'tempo_medio_duracao_alta_e_alta_efetiva' => 0,

                'tot_pacientes_dia' => 0,
                'tot_leitos_dia' => 0,

                'taxa_ocupacao' => 0,
                'tot_de_leitos' => $tot_de_leitos,

                'taxa_desocupacao' => 0,
                'intervalo_de_substituicao' => 0,

                'arr_idade_media' => [],
                'idade_media' => 0,

                'tot_paciente_sexo_masculino' => 0,
                'tot_paciente_sexo_feminino' => 0,

//'principais_diagnosticos' => [],
                //'principais_diagnosticos_sistemicos' => [],
                'principais_comorbidades' => [],
                'total_destino' => [],

                'tot_indice_giro_leito' => 0,
                'arr_leitos_utilizados' => [],
                'tempo_retorno' => [],

                'tempo_reinteracao' => [
                    '< 24h' => ['nome' => '24h', 'count' => 0],
                    '< 72h' => ['nome' => '72h', 'count' => 0],
                    '< 1 semana' => ['nome' => '1 semana', 'count' => 0],
                    '< 1 mês' => ['nome' => '1 mês', 'count' => 0],
                    '< 6 meses' => ['nome' => '6 meses', 'count' => 0],
                ],

                'faixa_internacao' => [
                    '< 24h' => ['nome' => '24h', 'count' => 0],
                    '2 a 7 dias' => ['nome' => '2 a 7 dias', 'count' => 0],
                    '8 a 14 dias' => ['nome' => '8 a 14 dias', 'count' => 0],
                    '15 a 21 dias' => ['nome' => '15 a 21 dias', 'count' => 0],
                    '21 a 28 dias' => ['nome' => '21 a 28 dias', 'count' => 0],
                    '> 28 dias' => ['nome' => '28 dias', 'count' => 0],
                ],

                '10_principais_origin_internacao' => [],

                '5_principais_origin_internacao' => [],

                'diagnostics' => $diagnostics_t,
                '15_principais_diagnosticos' => [],
                '5_principais_diagnosticos' => [],
                '5_principais_diagnosticos_sistemicos' => [],

                'systemic_diagnosis' => $systemic_diagnosis_formatted,
                '15_principais_diagnosticos_sis' => [],

                '15_principais_diagnosticos_sec' => [],

                '15_principais_comorbidades' => [],

                '5_principais_comorbidades' => [],

                'health_insurance_plans' => $health_insurance_plans_formatted,
                '20_principais_saude_complementares' => [],
                '5_principais_saude_complementares' => [],

                'arr_pacientes_saude_complementar' => [],

                'genero' => [
                    'm' => ['tipo' => 'Masculino', 'count' => 0, 'percentual' => 0],
                    'f' => ['tipo' => 'Feminino', 'count' => 0, 'percentual' => 0],
                ],

                //Tabela dos valores absolutos e percentuais frente ao total de internações com a distribuição das idades por faixa etária ( < 28 dias, 29 dias a 5 anos, 5 anos a 12, 12 a 18, 18 a 30, 31 a 40, 41 a 50, 51 a 60, 61 a 70, 71 a 80, 81 a 90 e > 90 anos
                'intenacao_faixa_etaria' => [
                    '1' => ['faixa_etaria' => '28 dias', 'count' => 0, 'percentual' => 0],
                    '2' => ['faixa_etaria' => '29 dias 5 anos', 'count' => 0, 'percentual' => 0],
                    '3' => ['faixa_etaria' => '6 a 12 anos', 'count' => 0, 'percentual' => 0],
                    '4' => ['faixa_etaria' => '13 a 18 anos', 'count' => 0, 'percentual' => 0],
                    '5' => ['faixa_etaria' => '19 a 30 anos ', 'count' => 0, 'percentual' => 0],
                    '6' => ['faixa_etaria' => '31 a 40 anos', 'count' => 0, 'percentual' => 0],
                    '7' => ['faixa_etaria' => '41 a 50 anos', 'count' => 0, 'percentual' => 0],
                    '8' => ['faixa_etaria' => '51 a 60 anos', 'count' => 0, 'percentual' => 0],
                    '9' => ['faixa_etaria' => '61 a 70 anos', 'count' => 0, 'percentual' => 0],
                    '10' => ['faixa_etaria' => '71 a 80 anos', 'count' => 0, 'percentual' => 0],
                    '11' => ['faixa_etaria' => '81 a 90 anos', 'count' => 0, 'percentual' => 0],
                    '12' => ['faixa_etaria' => '91 anos', 'count' => 0, 'percentual' => 0],
                ],
            ];

            foreach ($origins as $key => $origin)
            {
                $origin_name = $origin['name'];
                $data['sectors'][$sector_id]['arr_idade_obito_origem'][$origin_name]['arr'] = [];
                $data['sectors'][$sector_id]['arr_idade_obito_origem'][$origin_name]['media'] = 0;
                $data['sectors'][$sector_id]['arr_idade_obito_origem'][$origin_name]['count'] = 0;
                $data['sectors'][$sector_id]['arr_idade_obito_origem'][$origin_name]['nome'] = $origin['name'];

                $data['sectors'][$sector_id]['arr_idade_alta_origem'][$origin_name]['arr'] = [];
                $data['sectors'][$sector_id]['arr_idade_alta_origem'][$origin_name]['media'] = 0;
                $data['sectors'][$sector_id]['arr_idade_alta_origem'][$origin_name]['count'] = 0;
                $data['sectors'][$sector_id]['arr_idade_alta_origem'][$origin_name]['nome'] = $origin['name'];

                $data['sectors'][$sector_id]['arr_origem'][$origin_name]['count'] = 0;
                $data['sectors'][$sector_id]['arr_origem'][$origin_name]['name'] = $origin['name'];
                $data['sectors'][$sector_id]['arr_origem'][$origin_name]['percentual'] = 0;

                $data['sectors'][$sector_id]['arr_origem'][$origin_name]['systemic_diagnosis'] = $diagnostics_formatted;
                $data['sectors'][$sector_id]['arr_origem'][$origin_name]['diagnosticos_sis'] = [];

                $data['sectors'][$sector_id]['arr_origem'][$origin_name]['health_insurance'] = $health_insurance_plans_formatted;
            }
        }

        unset($diagnostics_t);
        unset($diagnostics_formatted);
        unset($systemic_diagnosis_formatted);

        foreach ($hospitalizations as $key_hospitalization => $hospitalization)
        {
//echo "<pre>";
            //print_r($hospitalization);
            $origin_id = null;
            $health_insurance_id = null;

            if (isset($hospitalization['historic_patient_movements'][0]))
            {
                $sector_id = $hospitalization['historic_patient_movements'][0]['room']['sector_id'];
            }

            if ($hospitalization['patient']['health_insurance_plan_id'])
            {
                $health_insurance = $hospitalization['patient']['health_insurance_plan_id'];
            }

            if ($hospitalization['origin_id'])
            {
                $origin_id = $hospitalization['origin_id'];
            }

            foreach ($hospitalization['discharges'] as $key => $discharge)
            {
                $data_saida_unidade = explode("/", $discharge['data_saida_unidade']);

                if (isset($discharge['hora_saida_unidade']))
                {
                    $data_s = $data_saida_unidade[0] . '-' . $data_saida_unidade[1] . '-' . $data_saida_unidade[2] . " " . $discharge['hora_saida_unidade']->i18nFormat('HH:mm');
                }
                else
                {
                    $data_s = $data_saida_unidade[0] . '-' . $data_saida_unidade[1] . '-' . $data_saida_unidade[2];
                }

                //echo $data_s;
                $data_saida_internacao = new \Datetime($data_s);

                $data['sectors'][$sector_id]['tempo_retorno'][$hospitalization['patient_id']]['tempo_saida'] = $data_saida_internacao;
            }

            if (isset($data['sectors'][$sector_id]['tempo_retorno'][$hospitalization['patient_id']]['tempo_saida']))
            {
                $data['sectors'][$sector_id]['tempo_retorno'][$hospitalization['patient_id']]['tempo_entrada'] = new \Datetime($hospitalization->date_of_hospitalization->i18nFormat('dd-MM-yyyy'));
            }

            if ($hospitalization['status'] == 'obito')
            {
                $data['tot_pacientes_com_obito']++;
            }
            elseif ($hospitalization['status'] == 'alta')
            {
                $data['tot_pacientes_com_alta']++;
            }

            if ($hospitalization['date_of_hospitalization'])
            {
                $date_of_hospitalization = new \DateTime($hospitalization['date_of_hospitalization']->format("Y-m-d"));

                if ($date_of_hospitalization >= $startDate &&
                    $date_of_hospitalization <= $endDate)
                {
                    $data['tot_internacoes_no_periodo']++;
                }
            }

//calc total de internacoes no periodo
            if (!empty($hospitalization['historic_patient_movements']))
            {
                $sector_id = $hospitalization['historic_patient_movements'][0]['room']['sector_id'];

                if ($origin_id && $health_insurance_id)
                {
                    $data['sectors'][$sector_id]['arr_origem'][$origin_id]['health_insurance'][$health_insurance_id]['tot']++;
                }

                if (!in_array($hospitalization->id, $data['sectors'][$sector_id]['arr_tot_internacoes']))
                {
                    array_push($data['sectors'][$sector_id]['arr_tot_internacoes'], $hospitalization->id);
                    $data['sectors'][$sector_id]['tot_internacoes_no_periodo']++;
                }

//ORIGEM DE INTERNAÇÃO ( 10 PRINCIPAIS + OUTROS )
                if ($hospitalization['origin_id'])
                {
                    $data['sectors'][$sector_id]['arr_origem'][$hospitalization['origin']['name']]['count']++;
                }

                if ($hospitalization['status'] == 'obito')
                {
                    $lastMovement = array_values(array_slice($hospitalization['historic_patient_movements'], -1))[0];
                    if (!in_array($hospitalization->id, $data['sectors'][$sector_id]['arr_tot_obito']))
                    {
                        array_push($data['sectors'][$sector_id]['arr_tot_obito'], $hospitalization->id);

//Media idade por obito
                        if (isset($hospitalization['patient']['birthday']))
                        {
                            $birthday = new \Datetime($hospitalization['patient']['birthday']->format('Y-m-d'));
                            $now = new \Datetime();
                            $age = $birthday->diff($now)->y;
                            array_push($data['sectors'][$sector_id]['arr_idade_obito'], $age);

                            if ($hospitalization['origin_id'])
                            {
                                array_push($data['sectors'][$sector_id]['arr_idade_obito_origem'][$hospitalization['origin']['name']]['arr'], $age);
                                $data['sectors'][$sector_id]['arr_idade_obito_origem'][$hospitalization['origin']['name']]['count']++;
                            }
                        }

                        $data['sectors'][$sector_id]['tot_obito']++;
                    }
                }
                elseif ($hospitalization['status'] == 'alta')
                {
                    $lastMovement = array_values(array_slice($hospitalization['historic_patient_movements'], -1))[0];

                    if (!in_array($hospitalization->id, $data['sectors'][$sector_id]['arr_tot_alta']))
                    {
                        array_push($data['sectors'][$sector_id]['arr_tot_alta'], $hospitalization->id);
                        $data['sectors'][$sector_id]['tot_alta']++;
                    }

//Media idade por alta
                    if (isset($hospitalization['patient']['birthday']))
                    {
                        $birthday = new \Datetime($hospitalization['patient']['birthday']->format('Y-m-d'));
                        $now = new \Datetime();
                        $age = $birthday->diff($now)->y;

                        array_push($data['sectors'][$sector_id]['arr_idade_alta'], $age);
                        if ($hospitalization['origin_id'])
                        {
                            array_push($data['sectors'][$sector_id]['arr_idade_alta_origem'][$hospitalization['origin']['name']]['arr'], $age);
                            $data['sectors'][$sector_id]['arr_idade_alta_origem'][$hospitalization['origin']['name']]['count']++;
                        }

                        $days = $birthday->diff($now)->days;
                        $tempo = $age;
                        if ($days <= 28 && $age == 0)
                        {
                            $tempo = '28dias';
                        }
                        elseif ($days > 28 && $age <= 5)
                        {
                            $tempo = 1;
                        }

                        if ($tempo == '28dias')
                        {
                            $data['sectors'][$sector_id]['intenacao_faixa_etaria'][1]['count']++;
                        }
                        elseif ($tempo >= 1 && $tempo <= 5)
                        {
                            $data['sectors'][$sector_id]['intenacao_faixa_etaria'][2]['count']++;
                        }
                        elseif ($tempo >= 6 && $tempo <= 12)
                        {
                            $data['sectors'][$sector_id]['intenacao_faixa_etaria'][3]['count']++;
                        }
                        elseif ($tempo >= 13 && $tempo <= 18)
                        {
                            $data['sectors'][$sector_id]['intenacao_faixa_etaria'][4]['count']++;
                        }
                        elseif ($tempo >= 19 && $tempo <= 30)
                        {
                            $data['sectors'][$sector_id]['intenacao_faixa_etaria'][5]['count']++;
                        }
                        elseif ($tempo >= 31 && $tempo <= 40)
                        {
                            $data['sectors'][$sector_id]['intenacao_faixa_etaria'][6]['count']++;
                        }
                        elseif ($tempo >= 41 && $tempo <= 50)
                        {
                            $data['sectors'][$sector_id]['intenacao_faixa_etaria'][7]['count']++;
                        }
                        elseif ($tempo >= 51 && $tempo <= 60)
                        {
                            $data['sectors'][$sector_id]['intenacao_faixa_etaria'][8]['count']++;
                        }
                        elseif ($tempo >= 61 && $tempo <= 70)
                        {
                            $data['sectors'][$sector_id]['intenacao_faixa_etaria'][9]['count']++;
                        }
                        elseif ($tempo >= 71 && $tempo <= 80)
                        {
                            $data['sectors'][$sector_id]['intenacao_faixa_etaria'][10]['count']++;
                        }
                        elseif ($tempo >= 81 && $tempo <= 90)
                        {
                            $data['sectors'][$sector_id]['intenacao_faixa_etaria'][11]['count']++;
                        }
                        elseif ($tempo > 90)
                        {
                            $data['sectors'][$sector_id]['intenacao_faixa_etaria'][12]['count']++;
                        }
                    }
                }

                foreach ($hospitalization['diagnostics'] as $diagnostic)
                {
                    if ($diagnostic['sector_id'] &&
                        $diagnostic['diagnostic_date'] >= $startDate &&
                        $diagnostic['diagnostic_date'] <= $endDate)
                    {
                        $sector_id = $diagnostic['sector_id'];
                    }
                    else
                    {
                        continue;
                    }

                    foreach ($diagnostic['systemic_diagnosis'] as $diag)
                    {
                        foreach ($data['sectors'][$sector_id]['diagnostics'] as $sector_diagnostic_key => $sector_diagnostic)
                        {
                            if ($sector_diagnostic['name'] == $diagnostic['description'])
                            {
                                $data['sectors'][$sector_id]['diagnostics'][$sector_diagnostic_key]['tot']++;

                                if ($diagnostic['type'] == 'Secundario')
                                {
                                    $data['sectors'][$sector_id]['diagnostics'][$sector_sec_diagnostic_key]['tot_secundario']++;
                                }
                            }
                        }
                    }

/*if (isset($data['sectors'][$sector_id]['principais_diagnosticos'][$diagnostic['description']]['count']))
{
$data['sectors'][$sector_id]['principais_diagnosticos'][$diagnostic['description']]['count']++;
}
else
{
$data['sectors'][$sector_id]['principais_diagnosticos'][$diagnostic['description']]['count'] = 1;
}

foreach ($diagnostic['systemic_diagnosis'] as $systemic_diagnosis)
{
if (isset($data['sectors'][$sector_id]['principais_diagnosticos_sistemicos'][$systemic_diagnosis['name']]['count']))
{
$data['sectors'][$sector_id]['principais_diagnosticos_sistemicos'][$systemic_diagnosis['name']]['count']++;
}
else
{
$data['sectors'][$sector_id]['principais_diagnosticos_sistemicos'][$systemic_diagnosis['name']]['count'] = 1;
}

foreach ($data['sectors'][$sector_id]['systemic_diagnosis'] as $sector_systemic_diagnostic_key => $sector_systemic_diagnostic)
{
if ($sector_systemic_diagnostic['name'] == $systemic_diagnosis['systemic_diagnosis'])
{
$data['sectors'][$sector_id]['systemic_diagnosis'][$sector_systemic_diagnostic_key]['tot_pacientes']++;
}
}
}*/

                    foreach ($diagnostic['comorbidities'] as $comorbidities)
                    {
                        if (isset($data['sectors'][$sector_id]['principais_comorbidades'][$comorbidities['name']]['count']))
                        {
                            $data['sectors'][$sector_id]['principais_comorbidades'][$comorbidities['name']]['count']++;
                        }
                        else
                        {
                            $data['sectors'][$sector_id]['principais_comorbidades'][$comorbidities['name']]['count'] = 1;
                        }

                        foreach ($data['sectors'][$sector_id]['diagnostics'] as $sector_diagnostic_key => $sector_diagnostic)
                        {
                            if ($sector_diagnostic['name'] == $comorbidities['name'])
                            {
                                $data['sectors'][$sector_id]['diagnostics'][$sector_diagnostic_key]['tot_comorbidade']++;
                            }
                        }
                    }
                }
            }

            foreach ($hospitalization['historic_patient_movements'] as $key_movimentacao => $movimentacao)
            {
                if ($movimentacao['room'])
                {
                    $sector_id = $movimentacao['room']['sector_id'];

                    if ($movimentacao['data_inicio'] && $movimentacao['data_termino'])
                    {
                        $date_start = new \DateTime($movimentacao['data_inicio']->format('Y-m-d'));
                        $date_end = new \DateTime($movimentacao['data_termino']->format('Y-m-d'));
                        $days = $date_start->diff($date_end)->days;

                        array_push($data['sectors'][$sector_id]['arr_tempo_medio_permanencia'], $hospitalization->id);

                        if ($days < 1)
                        {
                            $data['sectors'][$sector_id]['< 24h']['count']++;
                        }
                        elseif ($days >= 2 && $days <= 7)
                        {
                            $data['sectors'][$sector_id]['faixa_internacao']['2 a 7 dias']['count']++;
                        }
                        elseif ($days >= 8 && $days <= 14)
                        {
                            $data['sectors'][$sector_id]['faixa_internacao']['8 a 14 dias']['count']++;
                        }
                        elseif ($days >= 15 && $days <= 21)
                        {
                            $data['sectors'][$sector_id]['faixa_internacao']['15 a 21 dias']['count']++;
                        }
                        elseif ($days >= 22 && $days <= 28)
                        {
                            $data['sectors'][$sector_id]['faixa_internacao']['22 a 28 dias']['count']++;
                        }
                        elseif ($days > 28)
                        {
                            $data['sectors'][$sector_id]['faixa_internacao']['> 28 dias']['count']++;
                        }

                        if ($hospitalization->status == "obito")
                        {
                            array_push($data['sectors'][$sector_id]['arr_tempo_medio_permanencia_obito'], $days);
                        }

                        if ($hospitalization->status == "alta")
                        {
                            array_push($data['sectors'][$sector_id]['arr_tempo_medio_permanencia_alta'], $days);
                        }
                    }

//calc indice giro de leito
                    if (!in_array($movimentacao['room_id'], $data['sectors'][$sector_id]['arr_leitos_utilizados']))
                    {
                        array_push($data['sectors'][$sector_id]['arr_leitos_utilizados'], $movimentacao['room_id']);
                    }
                    else
                    {
                        $data['sectors'][$sector_id]['tot_indice_giro_leito']++;
                    }

                    if (isset($data['principais_destinos'][$sector_id]))
                    {
                        $data['principais_destinos'][$sector_id]['count']++;
                    }
                    else
                    {
                        $data['principais_destinos'][$sector_id]['nome'] = $movimentacao['room']['sector']['name'];
                        $data['principais_destinos'][$sector_id]['count'] = 1;
                    }

                    if (!in_array($hospitalization->id, $data['sectors'][$sector_id]['arr_pacientes_atendidos']))
                    {
                        array_push($data['sectors'][$sector_id]['arr_pacientes_atendidos'], $hospitalization->id);
                        $data['sectors'][$sector_id]['tot_pacientes_atendidos']++;

                        $birthday = new \Datetime($hospitalization['patient']['birthday']->format('Y-m-d'));
                        $now = new \Datetime();
                        $age = $birthday->diff($now)->y;

                        array_push($data['sectors'][$sector_id]['arr_idade_media'], $age);

                        if ($hospitalization['patient']['sexo'] == 'Masculino')
                        {
                            $data['sectors'][$sector_id]['tot_paciente_sexo_masculino']++;
                        }
                        else
                        {
                            $data['sectors'][$sector_id]['tot_paciente_sexo_feminino']++;
                        }
                    }

                    if (!empty($hospitalization['patient']['health_insurance_plan_id']))
                    {
                        if (!in_array($hospitalization->patient_id, $data['sectors'][$sector_id]['arr_pacientes_saude_complementar']))
                        {
                            array_push($data['sectors'][$sector_id]['arr_pacientes_saude_complementar'], $hospitalization->patient_id);
                            foreach ($data['sectors'][$sector_id]['health_insurance_plans'] as $key_health_insurance_plan => $health_insurance_plan)
                            {
                                if ($health_insurance_plan['name'] == $hospitalization->patient->health_insurance_plan->name)
                                {
                                    //echo $health_insurance_plan['name'];
                                    $data['sectors'][$sector_id]['health_insurance_plans'][$key_health_insurance_plan]['tot']++;
                                }
                            }
                        }

                        /*if (!in_array($hospitalization->id, $data['sectors'][$sector_id]['principais_convenios']['arr_pacientes']))
                    {
                    array_push($data['sectors'][$sector_id]['principais_convenios']['arr_pacientes'], $hospitalization->id);
                    if (isset($data['sectors'][$sector_id]['principais_convenios']['convenios'][$hospitalization['patient']['health_insurance_plan_id']]))
                    {
                    $data['sectors'][$sector_id]['principais_convenios']['convenios'][$hospitalization['patient']['health_insurance_plan_id']]['count']++;
                    }
                    else
                    {
                    $data['sectors'][$sector_id]['principais_convenios']['convenios'][$hospitalization['patient']['health_insurance_plan_id']]['nome'] = $hospitalization['patient']['health_insurance_plan']['name'];
                    $data['sectors'][$sector_id]['principais_convenios']['convenios'][$hospitalization['patient']['health_insurance_plan_id']]['count'] = 1;
                    }
                    }*/
                    }

                    if ($movimentacao['data_decisao_alta'])
                    {
                        if ($key_movimentacao == 0)
                        {
                            $date_decisao = new \DateTime($movimentacao['data_decisao_alta']->format('Y-m-d'));
                        }
                        elseif (isset($movimentacao['data_inicio']))
                        {

                            if (isset($data_desisao))
                            {
                                $date_start = new \DateTime($movimentacao['data_inicio']->format('Y-m-d'));
                                $intervalo_decisao = $date_decisao->diff($date_start);
                                //calculando horas
                                $horas = ($intervalo_decisao->days * 24) + $intervalo_decisao->h;

                                array_push($data['sectors'][$sector_id]['arr_tempo_medio_duracao_alta_e_alta_efetiva'], $horas);
                                $date_decisao = new \DateTime($movimentacao['data_decisao_alta']->format('Y-m-d'));
                            }
                            else
                            {
                                $date_decisao = new \DateTime($movimentacao['data_decisao_alta']->format('Y-m-d'));
                            }
                        }
                    }
                }
            }
        }

        foreach ($data['sectors'] as $key => $sector)
        {
            if (count($sector['arr_tempo_medio_permanencia']) > 0)
            {
                $data['sectors'][$key]['tempo_medio_permanencia'] = $calc_obj->calc_average($sector['arr_tempo_medio_permanencia']);
                $data['sectors'][$key]['tempo_minimo_permanencia'] = min($sector['arr_tempo_medio_permanencia']);
                $data['sectors'][$key]['tempo_maximo_permanencia'] = max($sector['arr_tempo_medio_permanencia']);
            }

            if (count($sector['arr_tempo_medio_permanencia_obito']) > 0)
            {
                $data['sectors'][$key]['tempo_medio_permanencia_obito'] = $calc_obj->calc_average($sector['arr_tempo_medio_permanencia_obito']);
            }

            if (count($sector['arr_tempo_medio_permanencia_alta']) > 0)
            {
                $data['sectors'][$key]['tempo_medio_permanencia_alta'] = $calc_obj->calc_average($sector['arr_tempo_medio_permanencia_alta']);
            }

            if (count($data['sectors'][$sector_id]['arr_tempo_medio_duracao_alta_e_alta_efetiva']) > 0)
            {
                $data['sectors'][$key]['tempo_medio_duracao_alta_e_alta_efetiva'] = $calc_obj->calc_average($sector['arr_tempo_medio_duracao_alta_e_alta_efetiva']);
            }

            if (count($sector['arr_idade_obito']) > 0)
            {
                $data['sectors'][$key]['media_idade_obito'] = $calc_obj->calc_average($sector['arr_idade_obito']);
            }

            if (count($sector['arr_idade_alta']) > 0)
            {
                $data['sectors'][$key]['media_idade_alta'] = $calc_obj->calc_average($sector['arr_idade_alta']);
            }

            if ($sector['tot_pacientes_atendidos'] > 0 && $sector['tot_obito'] > 0)
            {
                $data['sectors'][$key]['percen_obito'] = round(($sector['tot_obito'] * 100) / $sector['tot_pacientes_atendidos'], 2);
            }

            if ($sector['tot_pacientes_atendidos'] > 0 && $sector['tot_alta'] > 0)
            {
                $data['sectors'][$key]['percen_alta'] = round(($sector['tot_alta'] * 100) / $sector['tot_pacientes_atendidos'], 2);
            }

            $count_arr_pacientes_saude_complementar = count($sector['arr_pacientes_saude_complementar']);

            if ($count_arr_pacientes_saude_complementar > 0)
            {
                foreach ($sector['health_insurance_plans'] as $key_heath => $health_insurance_plan)
                {
                    $hearth_percent = ($health_insurance_plan['tot'] * 100) / $count_arr_pacientes_saude_complementar;
                    $data['sectors'][$key]['health_insurance_plans'][$key_heath]['percen'] = round($hearth_percent, 2);
                }
            }

            $tot_de_leitos_disponiveis = $sector['tot_de_leitos'] - $sector['tot_pacientes_atendidos'];

            $data['sectors'][$key]['tot_pacientes_dia'] = $total_dias_no_periodo * $sector['tot_pacientes_atendidos'];
            $data['sectors'][$key]['tot_leitos_dia'] = $total_dias_no_periodo * $tot_de_leitos_disponiveis;

            if ($data['sectors'][$key]['tot_leitos_dia'] > 0 && $data['sectors'][$key]['tot_pacientes_dia'] > 0)
            {
                $data['sectors'][$key]['taxa_ocupacao'] = ($data['sectors'][$key]['tot_pacientes_dia'] / $data['sectors'][$key]['tot_leitos_dia']) * 100;
                $data['sectors'][$key]['taxa_ocupacao'] = round($data['sectors'][$key]['taxa_ocupacao'], 2);

                //calc taxa de desocupacao
                $data['sectors'][$key]['taxa_desocupacao'] = 100 - $data['sectors'][$key]['taxa_ocupacao'];
            }

//calc intervalo substituicao
            if ($data['sectors'][$key]['taxa_desocupacao'] > 0 && $data['sectors'][$key]['tempo_medio_permanencia'] > 0)
            {
                $data['sectors'][$key]['intervalo_de_substituicao'] = ($data['sectors'][$key]['tempo_medio_permanencia'] * $data['sectors'][$key]['taxa_desocupacao']) / $data['sectors'][$key]['taxa_desocupacao'];
            }

            if (count($sector['arr_idade_media']) > 0)
            {
                $data['sectors'][$key]['idade_media'] = $calc_obj->calc_average($sector['arr_idade_media']);
            }

/*foreach ($sector['principais_convenios']['convenios'] as $key_convenio => $convenio)
{
$data['sectors'][$key]['principais_convenios']['total'][$convenio['count']][$key_convenio]['total'] = $convenio['count'];
$data['sectors'][$key]['principais_convenios']['total'][$convenio['count']][$key_convenio]['nome'] = $convenio['nome'];
$percentual_convenio = ($convenio['count'] * 100) / count($data['sectors'][$key]['principais_convenios']['arr_pacientes']);
$data['sectors'][$key]['principais_convenios']['total'][$convenio['count']][$key_convenio]['percentual'] = round($percentual_convenio, 2);
}*/

/*foreach ($sector['principais_diagnosticos'] as $key_diagnostico => $diagnostic)
{
$data['sectors'][$key]['principais_diagnosticos']['total'][$diagnostic['count']][$key_diagnostico]['nome'] = $key_diagnostico;
$data['sectors'][$key]['principais_diagnosticos']['total'][$diagnostic['count']][$key_diagnostico]['total'] = $diagnostic['count'];
$percentual_diagnostic = ($diagnostic['count'] * 100) / $data['sectors'][$key]['tot_pacientes_atendidos'];
$data['sectors'][$key]['principais_diagnosticos']['total'][$diagnostic['count']][$key_diagnostico]['percentual'] = round($percentual_diagnostic, 2);
}

foreach ($sector['principais_diagnosticos_sistemicos'] as $key_diagnostico => $diagnostic)
{
$data['sectors'][$key]['principais_diagnosticos_sistemicos']['total'][$diagnostic['count']][$key_diagnostico]['nome'] = $key_diagnostico;
$data['sectors'][$key]['principais_diagnosticos_sistemicos']['total'][$diagnostic['count']][$key_diagnostico]['total'] = $diagnostic['count'];
$percentual_diagnostic = ($diagnostic['count'] * 100) / $data['sectors'][$key]['tot_pacientes_atendidos'];
$data['sectors'][$key]['principais_diagnosticos_sistemicos']['total'][$diagnostic['count']][$key_diagnostico]['percentual'] = round($percentual_diagnostic, 2);
}*/

            foreach ($sector['principais_comorbidades'] as $key_comorbidade => $comorbidade)
            {
                $data['sectors'][$key]['principais_comorbidades']['total'][$comorbidade['count']][$key_comorbidade]['nome'] = $key_comorbidade;
                $data['sectors'][$key]['principais_comorbidades']['total'][$comorbidade['count']][$key_comorbidade]['total'] = $comorbidade['count'];
                $percentual_comorbidade = ($comorbidade['count'] * 100) / $data['sectors'][$key]['tot_pacientes_atendidos'];
                $data['sectors'][$key]['principais_comorbidades']['total'][$comorbidade['count']][$key_comorbidade]['percentual'] = round($percentual_comorbidade, 2);
            }

            foreach ($sector['arr_idade_obito_origem'] as $key_idade => $arr_idade)
            {
                $data['sectors'][$key]['arr_idade_obito_origem'][$key_idade]['media'] = $calc_obj->calc_average($arr_idade['arr']);
            }

            usort($data['sectors'][$key]['arr_idade_obito_origem'], function ($a, $b)
            {
                return $a['count'] < $b['count'];
            });

            foreach ($sector['arr_idade_alta_origem'] as $key_idade => $arr_idade)
            {
                $data['sectors'][$key]['arr_idade_alta_origem'][$key_idade]['media'] = $calc_obj->calc_average($arr_idade['arr']);
            }

            usort($data['sectors'][$key]['arr_idade_alta_origem'], function ($a, $b)
            {
                return $a['count'] < $b['count'];
            });

            foreach ($sector['arr_origem'] as $key_origem => $arr_origem)
            {
                if ($data['sectors'][$key]['tot_internacoes_no_periodo'] > 0)
                {
                    $percentual_origem = ($arr_origem['count'] * 100) / $data['sectors'][$key]['tot_internacoes_no_periodo'];
                    $data['sectors'][$key]['arr_origem'][$key_origem]['percentual'] = round($percentual_origem, 2);
                }
            }

            usort($data['sectors'][$key]['arr_origem'], function ($a, $b)
            {
                return $a['count'] < $b['count'];
            });

            $data['sectors'][$key]['10_principais_origin_internacao'] = $this->getFirstItensFromArray(10, $data['sectors'][$key]['arr_origem'], ['count', 'percentual']);

            $data['sectors'][$key]['5_principais_origin_internacao'] = $this->getFirstItensFromArray(5, $data['sectors'][$key]['arr_origem'], ['count', 'percentual']);

            foreach ($sector['genero'] as $key_genero => $genero)
            {
                if ($key_genero == 'm')
                {
                    $total_masculino = $data['sectors'][$key]['tot_paciente_sexo_masculino'];
                    $data['sectors'][$key]['genero']['m']['count'] = $total_masculino;
                    if ($data['sectors'][$key]['tot_pacientes_atendidos'] > 0)
                    {
                        $genero_percentual = ($total_masculino * 100) / $data['sectors'][$key]['tot_pacientes_atendidos'];
                        $data['sectors'][$key]['genero']['m']['percentual'] = round($genero_percentual, 2);
                    }
                }
                else
                {
                    $total_feminino = $data['sectors'][$key]['tot_paciente_sexo_feminino'];
                    $data['sectors'][$key]['genero']['f']['count'] = $total_feminino;
                    if ($data['sectors'][$key]['tot_pacientes_atendidos'] > 0)
                    {
                        $genero_percentual = ($total_feminino * 100) / $data['sectors'][$key]['tot_pacientes_atendidos'];
                        $data['sectors'][$key]['genero']['f']['percentual'] = round($genero_percentual, 2);
                    }
                }
            }

            foreach ($sector['intenacao_faixa_etaria'] as $key_i => $faixa_etaria)
            {
                $intenacao_faixa_etaria = $faixa_etaria['count'];
                if ($data['sectors'][$key]['tot_pacientes_atendidos'] > 0)
                {
                    $percentual = ($intenacao_faixa_etaria * 100) / $data['sectors'][$key]['tot_pacientes_atendidos'];
                    $data['sectors'][$key]['intenacao_faixa_etaria'][$key_i]['percentual'] = round($percentual, 2);
                }
            }
        }

        foreach ($data['principais_destinos'] as $key_destino => $destino)
        {
            $data['principais_destinos']['total'][$destino['count']][$key_destino]['name'] = $destino['nome'];
            $data['principais_destinos']['total'][$destino['count']][$key_destino]['total'] = $destino['count'];
            $percentual_destino = ($destino['count'] * 100) / $data['sectors'][$key]['tot_pacientes_atendidos'];
            $data['principais_destinos']['total'][$destino['count']][$key_destino]['percentual'] = round($percentual_destino, 2);
            # code...
        }

        foreach ($data['sectors'] as $key => $sector)
        {
            foreach ($data['principais_destinos'] as $key_destino => $value)
            {
                if ($key == $key_destino)
                {
                    $data['sectors'][$key]['total_destino'] = $value;
                }
            }
        }

        krsort($data['principais_destinos']['total']);

        foreach ($data['sectors'] as $key => $sector)
        {
            $count_origin = 0;

            foreach ($sector['tempo_retorno'] as $key_retorno => $retorno)
            {
//echo ("<pre>");
                //print_r($retorno);
                $intervalo = $retorno['tempo_saida']->diff($retorno['tempo_entrada']);

                if ($intervalo->days < 1)
                {
                    $data['sectors'][$key]['tempo_reinteracao']['< 24h']['count']++;
                }
                elseif ($intervalo->days < 3)
                {
                    $data['sectors'][$key]['tempo_reinteracao']['< 72h']['count']++;
                }
                elseif ($intervalo->days < 7)
                {
                    $data['sectors'][$key]['tempo_reinteracao']['< 1 semana']['count']++;
                }
                elseif ($intervalo->m < 1)
                {
                    $data['sectors'][$key]['tempo_reinteracao']['< 1 mês']['count']++;
                }
                elseif ($intervalo->m < 6)
                {
                    $data['sectors'][$key]['tempo_reinteracao']['< 6 meses']['count']++;
                }
            }

            usort($data['sectors'][$key]['diagnostics'], function ($a, $b)
            {
                return $a['tot'] < $b['tot'];
            });

            $data['sectors'][$key]['15_principais_diagnosticos'] = $this->getFirstItensFromArray(15, $data['sectors'][$key]['diagnostics'], ['tot', 'percen']);
            $data['sectors'][$key]['5_principais_diagnosticos'] = $this->getFirstItensFromArray(5, $data['sectors'][$key]['diagnostics'], ['tot', 'percen']);

            usort($data['sectors'][$key]['diagnostics'], function ($a, $b)
            {
                return $a['tot_comorbidade'] < $b['tot_comorbidade'];
            });

            $data['sectors'][$key]['15_principais_comorbidades'] = $this->getFirstItensFromArray(15, $data['sectors'][$key]['diagnostics'], ['tot_comorbidade', 'percen_comorbidade']);

            $data['sectors'][$key]['5_principais_comorbidades'] = $this->getFirstItensFromArray(5, $data['sectors'][$key]['diagnostics'], ['tot_comorbidade', 'percen_comorbidade']);

            usort($data['sectors'][$key]['diagnostics'], function ($a, $b)
            {
                return $a['tot_secundario'] < $b['tot_secundario'];
            });

            $data['sectors'][$key]['15_principais_diagnosticos_sec'] = $this->getFirstItensFromArray(15, $data['sectors'][$key]['diagnostics'], ['tot_secundario', 'percen_secundario']);

            usort($data['sectors'][$key]['systemic_diagnosis'], function ($a, $b)
            {
                return $a['tot_pacientes'] < $b['tot_pacientes'];
            });

            $data['sectors'][$key]['15_principais_diagnosticos_sis'] = $this->getFirstItensFromArray(15, $data['sectors'][$key]['systemic_diagnosis'], ['tot_pacientes', 'percen_pacientes']);

            $data['sectors'][$key]['5_principais_diagnosticos_sistemicos'] = $this->getFirstItensFromArray(5, $data['sectors'][$key]['systemic_diagnosis'], ['tot_pacientes', 'percen_pacientes']);

            usort($data['sectors'][$key]['health_insurance_plans'], function ($a, $b)
            {
                return $a['tot'] < $b['tot'];
            });

            $data['sectors'][$key]['20_principais_saude_complementares'] = $this->getFirstItensFromArray(20, $data['sectors'][$key]['health_insurance_plans'], ['tot', 'percen']);

            $data['sectors'][$key]['5_principais_saude_complementares'] = $this->getFirstItensFromArray(5, $data['sectors'][$key]['health_insurance_plans'], ['tot', 'percen']);

            foreach ($sector['arr_idade_obito_origem'] as $key_idade => $arr_idade)
            {
                if ($key_idade > 4)
                {
                    unset($data['sectors'][$key]['arr_idade_obito_origem'][$key_idade]);
                }
            }

            foreach ($sector['arr_idade_alta_origem'] as $key_idade => $arr_idade)
            {
                if ($key_idade > 4)
                {
                    unset($data['sectors'][$key]['arr_idade_alta_origem'][$key_idade]);
                }
            }
        }

        //echo memory_get_usage();
        return $data;
    }

    public function getRelatorioData($hospitalizations, $startOfMonth, $endOfMonth, $tot_de_leitos)
    {
        $lastDayOfMonth = new \DateTime($endOfMonth->format("Y/m/d"));
        $lastDayOfMonth->modify('last day of this month')->setTime(23, 59, 59);
        $lastDayOfMonth = (integer) $lastDayOfMonth->format("d");

        $tot_de_leitos_disponiveis = $tot_de_leitos;
        $calc_obj = new MedidasDispersao();

        $data = [
            'total_de_registros' => 0,
            'tot_leitos_dia' => 0,
            'tot_pacientes_dia' => 0,
            'arr_pacientes' => [

            ],
            'tot_pacientes' => 0,
            'total_de_internacoes' => 0, //total de  pacientes internados nesse mes
            'taxa_ocupacao' => 0,
            'taxa_desocupacao' => 0,
            'lastDayOfMonth' => $lastDayOfMonth,

            'indice_de_giro_de_leito' => 0,
            'tot_saidas' => 0,

            'tempo_medio_permanencia' => 0,

            'idade_media_arr' => [],
            'idade_media' => 0,

            'intervalo_de_substituicao' => 0,

            'arr_densidade_cateter_vesical' => [],
            'densidade_cateter_vesical' => 0,

            'arr_densidade_cateter_vascular' => [],
            'densidade_cateter_vascular' => 0,

            'tot_reinternacao' => 0,
            'taxa_de_reinternacao' => 0,

            'tot_reinternacao_menor_24' => 0,
            'faixa_de_internacao_menor_que_24h' => 0,

            'tot_obitos' => 0,
            'taxa_mortalidade' => 0,

            'tot_cesariana' => 0,
            'tot_cirurgia' => 0,
            'taxa_cesariana' => 0,

        ];

        $percentual = [];
        $i = 0;

        foreach ($hospitalizations as $hospitalization)
        {
            $i++;
            $data['total_de_registros']++;

//verifica quantos pacientes internados receberam alta
            switch ($hospitalization['status'])
            {
                case 'internado':
                    break;

                case 'obito':
                    $data['tot_saidas']++;
                    $data['tot_obitos']++;
                    break;

                case 'alta':
                    $data['tot_saidas']++;
                    break;
            }

//pega todas as novas internacoes desse periodo
            if ($hospitalization['date_of_hospitalization'] >= $startOfMonth &&
                $hospitalization['date_of_hospitalization'] <= $endOfMonth)
            {
                $data['total_de_internacoes']++;

//calcula a taxa de reinternacao
                if ($hospitalization['readmission_to_hospital_until_30_days'] ||
                    $hospitalization['rehospitalization_up_24'])
                {
                    $data['tot_reinternacao']++;
                }

                if ($hospitalization['rehospitalization_up_24'])
                {
                    $data['tot_reinternacao_menor_24']++;
                }
            }

            array_push($data['arr_pacientes'], $hospitalization->patient_id);

            foreach ($hospitalization['supportive_measures'] as $procedimento)
            {
//calcula os pacientes que fizeram uso de um dispositivo/procedimento
                if (stripos($procedimento['type'], 'cateter vesical') !== false)
                {
                    if ($procedimento['data_inicio'] >= $startOfMonth &&
                        $procedimento['data_inicio'] <= $endOfMonth
                    )
                    {
                        if (!in_array($hospitalization->id, $data['arr_densidade_cateter_vesical']))
                        {
                            array_push($data['arr_densidade_cateter_vesical'], $hospitalization->id);
                            $data['densidade_cateter_vesical']++;
                        }
                    }
                }

                if (stripos($procedimento['type'], 'cateter vascular') !== false)
                {
                    if ($procedimento['data_inicio'] >= $startOfMonth &&
                        $procedimento['data_inicio'] <= $endOfMonth
                    )
                    {
                        if (!in_array($hospitalization->id, $data['arr_densidade_cateter_vascular']))
                        {
                            array_push($data['arr_densidade_cateter_vascular'], $hospitalization->id);
                            $data['densidade_cateter_vascular']++;
                        }
                    }
                }
            }

            foreach ($hospitalization['historic_patient_movements'] as $movimentacao)
            {
//calcula os pacientes que foram transferidos de setor
                if (isset($movimentacao['data_inicio']))
                {
                    if ($movimentacao['data_inicio'] >= $startOfMonth &&
                        $movimentacao['data_inicio'] <= $endOfMonth
                    )
                    {
                        $data['tot_saidas']++;
                    }
                }
            }

//o certo seria pegar toas as cirurgias da unidade independente da internacao
            foreach ($hospitalization['surgeries'] as $cirurgia)
            {
//calcula as cirurgias do tipo cesariana
                if ($cirurgia['data_da_cirurgia'] >= $startOfMonth &&
                    $cirurgia['data_da_cirurgia'] <= $endOfMonth
                )
                {
                    if (stripos($cirurgia['nome_da_cirurgia'], 'cesariana') !== false)
                    {
                        $data['tot_cesariana']++;
                    }

                    $data['tot_cirurgia']++;
                }
            }
        }

        if (count($data['arr_pacientes']) > 0)
        {
            $data['tot_pacientes'] = count($data['arr_pacientes']);
            $tot_de_leitos_disponiveis -= $data['tot_pacientes'];

            $data['tot_pacientes_dia'] = $lastDayOfMonth * $data['tot_pacientes'];
            $data['tot_leitos_dia'] = $lastDayOfMonth * $tot_de_leitos_disponiveis;
        }

        if ($data['tot_pacientes_dia'] > 0 && $data['tot_leitos_dia'] > 0)
        {
            //calc taxa de ocupacao
            $data['taxa_ocupacao'] = ($data['tot_pacientes_dia'] / $data['tot_leitos_dia']) * 100;
            $data['taxa_ocupacao'] = round($data['taxa_ocupacao'], 2);

            //calc taxa de desocupacao
            $data['taxa_desocupacao'] = 100 - $data['taxa_ocupacao'];

//calc tempo de permanencia
            if ($data['tot_saidas'] > 0)
            {
                $data['tempo_medio_permanencia'] = round($data['tot_pacientes_dia'] / $data['tot_saidas'], 2);

                //calc indice do giro de leito
                $data['indice_de_giro_de_leito'] = round($data['tot_saidas'] / $tot_de_leitos_disponiveis, 2);
            }

//calc intervalo substituicao
            if ($data['taxa_desocupacao'] > 0 || $data['tempo_medio_permanencia'] > 0)
            {
                $data['intervalo_de_substituicao'] = ($data['tempo_medio_permanencia'] * $data['taxa_desocupacao']) / $data['taxa_desocupacao'];
            }
        }

//calc taxa reinternacao
        if ($data['total_de_internacoes'] > 0 &&
            $data['tot_reinternacao'] > 0)
        {
            $data['taxa_de_reinternacao'] = ($data['tot_reinternacao'] * 100) / $data['total_de_internacoes'];
            $data['taxa_de_reinternacao'] = round($data['taxa_de_reinternacao'], 2);
        }

//calc taxa reinternacao < 24h
        if ($data['total_de_internacoes'] > 0 &&
            $data['tot_reinternacao_menor_24'] > 0)
        {
            $data['faixa_de_internacao_menor_que_24h'] = ($data['tot_reinternacao_menor_24'] * 100) / $data['total_de_internacoes'];
            $data['faixa_de_internacao_menor_que_24h'] = round($data['faixa_de_internacao_menor_que_24h'], 2);
        }

//calc taxa de mortalidade
        if ($data['tot_saidas'] > 0 && $data['tot_obitos'] > 0)
        {
            $data['taxa_mortalidade'] = $data['tot_obitos'] / $data['tot_saidas'];
            $data['taxa_mortalidade'] = round($data['taxa_mortalidade'] * 100, 2);
        }

//calc taxa de cesariana
        if ($data['tot_cirurgia'] > 0 && $data['tot_cesariana'] > 0)
        {
            $data['taxa_cesariana'] = $data['tot_cesariana'] / $data['tot_cirurgia'];
            $data['taxa_cesariana'] = round($data['taxa_cesariana'] * 100, 2);
        }

        return $data;
    }

    public function hospitalizationsPacients($totalHospitalizationPatients)
    {
    }

    public function returnDiagnostics()
    {
        $cid10_table = TableRegistry::get('Cid10');
        $diagnostics = $cid10_table->find('all', [
            'order' => ['Cid10.cid_cod' => 'ASC'],
        ])
            ->toArray();
        //$diagnostics_formatted = [];
        $d_t = [];

        foreach ($diagnostics as $diagnostic)
        {
            $d_t[$diagnostic['id']] = [
                'name' => $diagnostic['description'],
                'tot' => 0,
                'percen' => 0,

                'tot_secundario' => 0,
                'percen_secundario' => 0,

                'tot_comorbidade' => 0,
                'percen_comorbidade' => 0,
            ];

            //array_push($diagnostics_formatted, $diagnostic_arr);
        }

        return $d_t;
    }

    public function getFirstItensFromArray($quantity, $array, $indexes)
    {
        $count = 0;

//$array['outras'] = $indexes;

        foreach ($indexes as $index_key => $index)
        {
            $array['outras'][$index] = 0;
            $array['outras']['name'] = 'Outros';
        }

        foreach ($array as $array_key => $value)
        {
            if ($count >= $quantity)
            {
                foreach ($indexes as $index)
                {
                    $array['outras'][$index] += $value[$index];
                }

                if ($array_key != 'outras')
                {
                    unset($array[$array_key]);
                }
            }

            $count++;
        }

        return $array;
    }

    // tev clinico
    public function hospitalizationsProtocolsTev($hospitalizationsProtocolsTev)
    {
        $countProtocoloTev = 0;

        foreach ($hospitalizationsProtocolsTev as $key => $value)
        {
            $countProtocoloTev += count($value['clinic_tev_protocols']);
        }

        return $countProtocoloTev;
    }

    // tev cirurgico
    public function hospitalizationsProtocolsTevCicurgico($data, $hospitalizationsProtocolsTev)
    {

        foreach ($hospitalizationsProtocolsTev as $key => $value)
        {
            if (!empty($value['surgical_tev_protocols']))
            {
                $count_surgical_protocolo = count($value['surgical_tev_protocols']);

                foreach ($value['surgical_tev_protocols'] as $key => $surgical_protocols)
                {
                    if ($key == ($count_surgical_protocolo - 1))
                    {
                        $data['paciente_protocolo_tev']['count_pacientes_protocolo_tev']++;

                        if (isset($data['paciente_protocolo_tev']['setor'][$value['sector_id']]['name']))
                        {
                            $data['paciente_protocolo_tev']['setor'][$value['sector_id']]['count'] += 1;
                        }
                        else
                        {
                            $data['paciente_protocolo_tev']['setor'][$value['sector_id']]['name'] = $value['sector']['name'];
                            $data['paciente_protocolo_tev']['setor'][$value['sector_id']]['count'] = 1;
                        }
                    }
                }
            }
        }

        return $data;
    }

    public function getSystemicDiagnosis()
    {
        $diagnosis = [
            "Cardiovascular",
            "Fígado Vias Biliares e Pâncreas",
            "Oncológicos",
            "Choque (exceto sepse)",
            "Gastrintestinal",
            "Osteoarticular",
            "Cuidados paliativos ",
            "Hematológico",
            "Outras",
            "Disfunção de Múltiplos Órgãos",
            "Infecção/Sepse",
            "Pós-PCR",
            "Doenças alérgicas auto-imunes e colagenoses",
            "Monitoração ",
            "Renal",
            "Doenças dos órgãos genitais femininos",
            "Morte encefálica ",
            "Respiratório (exceto sepse/infecção)",
            "Doenças dos órgãos genitais masculinos",
            "Neurológico ",
            "Trauma não-cirurgico",
            "Endócrino/Metabólico",
            "Obstétricos",
            "Cirurgia bariátrica",
            "Cirurgia do intestino delgado",
            "Glândulas endócrinas",
            "Cirurgia do esôfago",
            "Cirurgia ginecológica / mama",
            "Neurocirurgia",
            "Cirurgia gástrica",
            "Cirurgia torácica ",
            "outras ",
            "Obstétricos",
            "Cirurgia ortopédica",
            "Cirurgia abdominal / retroperitôneo ",
            "Outras cirurgias eletivas",
            "Cirurgia próstata",
            "Cirurgia cardíaca - cardiopatias Congênitas",
            "Pele e tecidos moles",
            "Cirurgia vascular",
            "Cirurgia cardíaca (exceto cardiopatia congênita)",
            "Politrauma Ciúrgico",
            "Cirurgia da via urinária",
            "Cirurgia oftalmológica ",
            "Procedimento endovascular",
            "Cirurgia - pulmão, traqueia e brônquios",
            "Cirurgia otorrinolaringológica ",
            "Procedimento invasivo (exceto cardiovasculares)",
            "Cirurgia de cabeça e pescoço",
            "Correção de hérnias e defeitos da parede abdominal ",
            "Procedimento invasivo cardíaco",
            "Cirurgia de coluna Fígado / Vias biliares / Pâncreas ",
            "Transplantes de órgãos sólidos",
            "Cirurgia do cólon / sigmóide / reto / canal anal",
            "Cirurgia do intestino delgado",
            "Glândulas endócrinas",
            "Cirurgia do esôfago",
            "Cirurgia ginecológica / mama",
            "Neurocirurgia",
            "Cirurgia gástrica",
            "Cirurgia torácica ",
            "outras ",
            "Obstétricos",
            "Cirurgia ortopédica",
            "Cirurgia abdominal / retroperitôneo ",
            "Outras cirurgias eletivas",
            "Cirurgia próstata",
            "Cirurgia cardíaca - cardiopatias Congênitas",
            "Pele e tecidos moles",
            "Cirurgia vascular",
            "Cirurgia cardíaca (exceto cardiopatia congênita)",
            "Politrauma Ciúrgico",
            "Cirurgia da via urinária",
            "Cirurgia oftalmológica ",
            "Procedimento endovascular",
            "Cirurgia - pulmão, traqueia e brônquios",
            "Cirurgia otorrinolaringológica ",
            "Procedimento invasivo (exceto cardiovasculares)",
            "Cirurgia de cabeça e pescoço",
            "Correção de hérnias e defeitos da parede abdominal ",
            "Procedimento invasivo cardíaco",
            "Cirurgia de coluna Fígado / Vias biliares / Pâncreas ",
            "Transplantes de órgãos sólidos",
            "Cirurgia do cólon / sigmóide / reto / canal anal",
            "Cardiopatias congênitas",
            "Endócrino-metabólicas ",
            "Osteo-articular",
            "Cardiovasculares (exceto cardiopatias congênitas) ",
            "Gastrintestinais ",
            "Outras complicações não-especificadas",
            "Choque (exceto sepse) ",
            "Hematológicas ",
            "Pós-PCR",
            "Complicações Maternas e Relacionadas à Gestação ",
            "Hepático ",
            "Prematuridade",
            "Complicações perinatais ",
            "Infecções ",
            "Renais",
            "Cuidados paliativos ",
            "Monitoração ",
            "Respiratórias",
            "Disfunção de Múltiplos Órgãos ",
            "Morte encefálica ",
            "Sistema Nervoso Central",
            "Doenças alérgicas, auto-imunes e colagenoses ",
            "Oncológicos ",
            "Trauma não-cirurgico",
            "Doenças dos órgãos genitais masculinos",
        ];

        return $diagnosis;
    }

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules)
    {
        $rules->add($rules->existsIn(['patient_id'], 'Patients'));
        $rules->add($rules->existsIn(['screening_id'], 'Screenings'));
        $rules->add($rules->existsIn(['origin_id'], 'Origins'));
        $rules->add($rules->existsIn(['unit_id'], 'Units'));
        $rules->add($rules->existsIn(['sector_id'], 'Sectors'));
        $rules->add($rules->existsIn(['medical_care_id'], 'MedicalCares'));
        return $rules;
    }
}
