<?php
namespace App\Model\Table;

use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;
use MedidasDispersao\MedidasDispersao;

require_once ROOT . DS . "vendor" . DS . "Zurubabel" . DS . "MedidasDispersao.php";

/**
 * Surgeries Model
 *
 * @property \Cake\ORM\Association\BelongsTo $Hospitalizations
 * @property \Cake\ORM\Association\BelongsToMany $Professionals
 * @property \Cake\ORM\Association\BelongsToMany $Specialties
 * @property \Cake\ORM\Association\BelongsToMany $SurgicalComplications
 * @property \Cake\ORM\Association\BelongsToMany $SurgicalMaterials
 * @property \Cake\ORM\Association\BelongsToMany $TypeOfAnesthesias
 */
class SurgeriesTable extends Table
{

    /**
     * Initialize method
     *
     * @param array $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config)
    {
        parent::initialize($config);

        $this->table('surgeries');
        $this->displayField('id');
        $this->primaryKey('id');

        $this->addBehavior('Timestamp');

        $this->belongsTo('Hospitalizations', [
            'foreignKey' => 'hospitalization_id',
            'joinType' => 'INNER',
        ]);
        $this->belongsTo('Rooms', [
            'foreignKey' => 'room_id',
            'joinType' => 'LEFT',
        ]);
        $this->belongsToMany('Professionals', [
            'foreignKey' => 'surgery_id',
            'targetForeignKey' => 'professional_id',
            'joinTable' => 'professionals_surgeries',
        ]);
        $this->belongsToMany('Specialties', [
            'foreignKey' => 'surgery_id',
            'targetForeignKey' => 'specialty_id',
            'joinTable' => 'specialties_surgeries',
        ]);
        $this->belongsToMany('AuxiliarySpecialties', [
            'foreignKey' => 'surgery_id',
            'targetForeignKey' => 'auxiliary_specialty_id',
            'joinTable' => 'auxiliary_specialties_surgeries',
        ]);
        $this->belongsToMany('ChangeSpecialties', [
            'foreignKey' => 'surgery_id',
            'targetForeignKey' => 'change_specialty_id',
            'joinTable' => 'change_specialties_surgeries',
        ]);
        $this->belongsToMany('SurgicalComplications', [
            'foreignKey' => 'surgery_id',
            'targetForeignKey' => 'surgical_complication_id',
            'joinTable' => 'surgeries_surgical_complications',
        ]);
        $this->belongsToMany('SurgicalMaterials', [
            'foreignKey' => 'surgery_id',
            'targetForeignKey' => 'surgical_material_id',
            'joinTable' => 'surgical_materials_surgeries',
        ]);
        $this->belongsToMany('TypeOfAnesthesias', [
            'foreignKey' => 'surgery_id',
            'targetForeignKey' => 'type_of_anesthesia_id',
            'joinTable' => 'type_of_anesthesias_surgeries',
        ]);
        $this->belongsToMany('ChangeTypeOfAnesthesias', [
            'foreignKey' => 'surgery_id',
            'targetForeignKey' => 'change_type_of_anesthesia_id',
            'joinTable' => 'change_type_of_anesthesias_surgeries',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator)
    {
        $validator
            ->integer('id')
            ->allowEmpty('id', 'create');

        $validator
            ->allowEmpty('nome');

        $validator
            ->allowEmpty('especialidade');

        $validator
            ->boolean('outra_especialidade_cirurgica_em_apoio')
            ->allowEmpty('outra_especialidade_cirurgica_em_apoio');

        $validator
            ->allowEmpty('especialidade_cirurgica_auxiliar');

        $validator
            ->allowEmpty('porte_da_cirurgia');

        $validator
            ->allowEmpty('caracteristica_da_cirurgia');

        $validator
            ->allowEmpty('cirurgia_contaminada');

        $validator
            ->boolean('anestesia_adequada_foi_confirmada')
            ->allowEmpty('anestesia_adequada_foi_confirmada');

        $validator
        //->date('data_da_solicitacao')
        ->allowEmpty('data_da_solicitacao');

        $validator
        //->date('data_da_cirurgia')
        ->allowEmpty('data_da_cirurgia');

        $validator
        //->date('data_da_liberacao')
        ->allowEmpty('data_da_liberacao');

        $validator
        //->time('tempo_de_cirurgia_estimado')
        ->allowEmpty('tempo_de_cirurgia_estimado');

        $validator
        //->time('hora_da_cirurgia')
        ->allowEmpty('hora_da_cirurgia');

        $validator
            ->boolean('internacao_previa_a_cirurgia')
            ->allowEmpty('internacao_previa_a_cirurgia');

        $validator
            ->boolean('deseja_reservar_leito')
            ->allowEmpty('deseja_reservar_leito');

        $validator
            ->integer('total_dias_do_leito_reservado')
            ->allowEmpty('total_dias_do_leito_reservado');

        $validator
            ->boolean('pos_operatorio_em_uti')
            ->allowEmpty('pos_operatorio_em_uti');

        $validator
            ->boolean('conversao_da_caracteristica_da_cirurgia')
            ->allowEmpty('conversao_da_caracteristica_da_cirurgia');

        $validator
            ->allowEmpty('conversao_para');

        $validator
            ->boolean('adiar_cirurgia')
            ->allowEmpty('adiar_cirurgia');

        $validator
        //->date('aidar_cirurgia_para_data')
        ->allowEmpty('aidar_cirurgia_para_data');

        $validator
            ->time('aidar_cirurgia_para_hora')
            ->allowEmpty('aidar_cirurgia_para_hora');

        $validator
            ->allowEmpty('adiar_cirurgia_motivo');

        $validator
            ->boolean('suspender_cirurgia')
            ->allowEmpty('suspender_cirurgia');

        $validator
        //->date('suspender_cirurgia_para_data')
        ->allowEmpty('suspender_cirurgia_para_data');

        $validator
            ->time('suspender_cirurgia_para_hora')
            ->allowEmpty('suspender_cirurgia_para_hora');

        $validator
            ->allowEmpty('suspender_cirurgia_motiv');

        $validator
            ->boolean('cancelar_cirurgia')
            ->allowEmpty('cancelar_cirurgia');

        $validator
            ->allowEmpty('cancelar_cirurgia_motivo');

        $validator
            ->boolean('visita_pre_anestesica')
            ->allowEmpty('visita_pre_anestesica');

        $validator
        //->date('visita_pre_anestesica_data')
        ->allowEmpty('visita_pre_anestesica_data');

        $validator
            ->time('visita_pre_anestesica_hora')
            ->allowEmpty('visita_pre_anestesica_hora');

        $validator
            ->boolean('entrada_na_sala_cirurgica')
            ->allowEmpty('entrada_na_sala_cirurgica');

        $validator
        //->date('entrada_na_sala_cirurgica_data')
        ->allowEmpty('entrada_na_sala_cirurgica_data');

        $validator
            ->time('entrada_na_sala_cirurgica_hora')
            ->allowEmpty('entrada_na_sala_cirurgica_hora');

        $validator
            ->boolean('realizado_antibiotico_profilaxia')
            ->allowEmpty('realizado_antibiotico_profilaxia');

        $validator
        //->date('realizado_antibiotico_profilaxia_data')
        ->allowEmpty('realizado_antibiotico_profilaxia_data');

        $validator
            ->time('realizado_antibiotico_profilaxia_hora')
            ->allowEmpty('realizado_antibiotico_profilaxia_hora');

        $validator
            ->time('inicio_da_anestesia_hora')
            ->allowEmpty('inicio_da_anestesia_hora');

        $validator
        //->time('termino_da_anestesia_hora')
        ->allowEmpty('termino_da_anestesia_hora');

        $validator
            ->time('inicio_da_cirurgia_hora')
            ->allowEmpty('inicio_da_cirurgia_hora');

        $validator
            ->time('termino_da_cirurgia_hora')
            ->allowEmpty('termino_da_cirurgia_hora');

        $validator
            ->boolean('houve_mudanca_na_cirurgia')
            ->allowEmpty('houve_mudanca_na_cirurgia');

        $validator
            ->allowEmpty('mudanca_na_cirurgia');

        $validator
            ->boolean('houve_mudanca_na_anestesia')
            ->allowEmpty('houve_mudanca_na_anestesia');

        $validator
            ->allowEmpty('mudanca_na_anestesia');

        $validator
            ->allowEmpty('houve_complicacao_durante_cirurgia');

        $validator
            ->boolean('uso_de_compressa_cirurgica')
            ->allowEmpty('uso_de_compressa_cirurgica');

        $validator
            ->integer('total_de_compressa_cirurgica_usada')
            ->allowEmpty('total_de_compressa_cirurgica_usada');

        $validator
            ->boolean('uso_de_pinca_extra')
            ->allowEmpty('uso_de_pinca_extra');

        $validator
            ->integer('total_de_pinca_extra_usada')
            ->allowEmpty('total_de_pinca_extra_usada');

        $validator
            ->boolean('uso_de_caixa_extra')
            ->allowEmpty('uso_de_caixa_extra');

        $validator
            ->integer('total_de_caixa_extra')
            ->allowEmpty('total_de_caixa_extra');

        $validator
            ->boolean('encaminhado_para_setor_de_destino_programado')
            ->allowEmpty('encaminhado_para_setor_de_destino_programado');

        $validator
            ->allowEmpty('setor_nao_programado');

        $validator
            ->allowEmpty('encaminhado_para_setor_nao_programado_motivo');

        $validator
            ->boolean('houve_reabordagem_cirurgica')
            ->allowEmpty('houve_reabordagem_cirurgica');

        $validator
            ->allowEmpty('reabordagem_programada');

        $validator
            ->boolean('retirada_de_ortese_ou_protese')
            ->allowEmpty('retirada_de_ortese_ou_protese');

        $validator
            ->allowEmpty('retirada_de_ortese_ou_protese_motivo');

        $validator
        //->date('data_da_instalacao_ortese_ou_protese')
        ->allowEmpty('data_da_instalacao_ortese_ou_protese');

        $validator
            ->allowEmpty('status');

        $validator
            ->boolean('risco_cirurgico')
            ->allowEmpty('risco_cirurgico');

        $validator
        //->date('data_risco_cirurgico')
        ->allowEmpty('data_risco_cirurgico');

        $validator
            ->allowEmpty('complicacao_durante_cirurgia');

        $validator
            ->time('hora_encaminhamento_rpa')
            ->allowEmpty('hora_encaminhamento_rpa');

        return $validator;
    }

    public function getRelatorioData($hospitalizations, $date_before, $date_after, $type_of_anesthesias, $specialties)
    {
        $calc_obj = new MedidasDispersao();

        $data = [
            'total_de_cirurgias' => 0,

            'total_de_cirurgias_contaminadas' => 0,
            'taxa_de_cirurgias_contaminadas' => 0,
            'anestesias' => [],
            'ids' => [],
            'surgeries' => [],
            'turnos' => [
                'manha' => [
                    'tot_cirurgias' => 0,

                    'taxa_anti_60m_antes_cirurgia' => 0,
                    'tot_anti_60m_antes_cirurgia' => 0,

                    'arr_tempo_realizacao_anestesia' => [],
                    'media_tempo_realizacao_anestesia' => 0,
                ],

                'tarde' => [
                    'tot_cirurgias' => 0,

                    'taxa_anti_60m_antes_cirurgia' => 0,
                    'tot_anti_60m_antes_cirurgia' => 0,

                    'arr_tempo_realizacao_anestesia' => [],
                    'media_tempo_realizacao_anestesia' => 0,
                ],

                'noite' => [
                    'tot_cirurgias' => 0,

                    'taxa_anti_60m_antes_cirurgia' => 0,
                    'tot_anti_60m_antes_cirurgia' => 0,

                    'arr_tempo_realizacao_anestesia' => [],
                    'media_tempo_realizacao_anestesia' => 0,
                ],

                'madrugada' => [
                    'tot_cirurgias' => 0,

                    'taxa_anti_60m_antes_cirurgia' => 0,
                    'tot_anti_60m_antes_cirurgia' => 0,

                    'arr_tempo_realizacao_anestesia' => [],
                    'media_tempo_realizacao_anestesia' => 0,
                ],
            ],
            'main_specialties' => [],
            'specialties' => [],
            'tot_cirurgia_ortese_protese' => 0,
            'tot_infeccao_ortese_protese' => 0,

            'taxa_infeccao_ortese_protese' => 0,

            'taxa_anti_60m_antes_cirurgia' => 0,
            'tot_anti_60m_antes_cirurgia' => 0,

            'taxa_anti_60m_antes_cirurgia_porte_pequeno' => 0,
            'tot_anti_60m_antes_cirurgia_porte_pequeno' => 0,

            'taxa_anti_60m_antes_cirurgia_porte_medio' => 0,
            'tot_anti_60m_antes_cirurgia_porte_medio' => 0,

            'taxa_anti_60m_antes_cirurgia_porte_grande' => 0,
            'tot_anti_60m_antes_cirurgia_porte_grande' => 0,

            
            'tot_anti_90m_antes_cirurgia' => 0,
            'taxa_anti_90m_antes_cirurgia' => 0,

            'tot_porte_da_cirurgia_pequeno' => 0,

            'tot_porte_da_cirurgia_medio' => 0,

            'tot_porte_da_cirurgia_grande' => 0,

            'dia_da_semana' => [
                0 => [                
                    'tot_cirurgias' => 0,
                    'taxa_anti_60m_antes_cirurgia' => 0,
                    'tot_anti_60m_antes_cirurgia' => 0,
                ],
                1 => [
                    'tot_cirurgias' => 0,
                    'taxa_anti_60m_antes_cirurgia' => 0,
                    'tot_anti_60m_antes_cirurgia' => 0,
                ],
                2 => [
                    'tot_cirurgias' => 0,
                    'taxa_anti_60m_antes_cirurgia' => 0,
                    'tot_anti_60m_antes_cirurgia' => 0,
                ],
                3 => [ 
                    'tot_cirurgias' => 0,
                    'taxa_anti_60m_antes_cirurgia' => 0,
                    'tot_anti_60m_antes_cirurgia' => 0,
                ],
                4 => [
                    'tot_cirurgias' => 0,
                    'taxa_anti_60m_antes_cirurgia' => 0,
                    'tot_anti_60m_antes_cirurgia' => 0,
                ],
                5 => [
                    'tot_cirurgias' => 0,
                    'taxa_anti_60m_antes_cirurgia' => 0,
                    'tot_anti_60m_antes_cirurgia' => 0,
                ],
                6 => [
                    'tot_cirurgias' => 0,
                    'taxa_anti_60m_antes_cirurgia' => 0,
                    'tot_anti_60m_antes_cirurgia' => 0,
                ],
            ],

            'arr_tempo_realizacao_anestesia' => [],
            'media_tempo_realizacao_anestesia' => 0,
        ];

        $columns = $this->schema()->columns();

        foreach ($columns as $key => $column)
        {
            $data[$column]['total'] = 0;
            $data[$column]['percentual'] = 0;
            $data[$column]['media'] = 0;
            $data[$column]['mediana'] = 0;
            $data[$column]['desvio_padrao'] = 0;
            $data[$column]['soma'] = 0;
        }

        $specialties_arr = [];

        foreach ($specialties as $specialty)
        {
            $specialty_id = $specialty['id'];

            $specialties_arr[$specialty_id] = [
                'id' => $specialty_id,
                'name' => $specialty['name'],
                'tot' => 0,
                'percen' => 0,

                'tot_cirurgias' => 0,

                'taxa_anti_60m_antes_cirurgia' => 0,
                'tot_anti_60m_antes_cirurgia' => 0,

                'arr_tempo_realizacao_anestesia' => [],
                'media_tempo_realizacao_anestesia' => 0,

                'turnos' => [
                    'manha' => [
                        'arr_tempo_realizacao_anestesia' => [],
                        'media_tempo_realizacao_anestesia' => 0,
                    ],

                    'tarde' => [
                        'arr_tempo_realizacao_anestesia' => [],
                        'media_tempo_realizacao_anestesia' => 0,
                    ],

                    'noite' => [
                        'arr_tempo_realizacao_anestesia' => [],
                        'media_tempo_realizacao_anestesia' => 0,
                    ],

                    'madrugada' => [
                        'arr_tempo_realizacao_anestesia' => [],
                        'media_tempo_realizacao_anestesia' => 0,
                    ],
                ],
            ];
        }

        $data['main_specialties'] = $specialties_arr;
        $data['specialties'] = $specialties_arr;


        foreach ($type_of_anesthesias as $anesthesia)
        {
            $anesthesia_id = $anesthesia['id'];
            $anesthesia_name = $anesthesia['nome'];
            $data['anestesias'][$anesthesia_id] = [
                'name' => $anesthesia_name,
                'total' => 0,

                'tot_cirurgias' => 0,

                'tot_porte_da_cirurgia_pequeno' => 0,
                'percen_porte_da_cirurgia_pequeno' => 0,

                'tot_porte_da_cirurgia_medio' => 0,
                'percen_porte_da_cirurgia_medio' => 0,

                'tot_porte_da_cirurgia_grande' => 0,
                'percen_porte_da_cirurgia_grande' => 0,

                'tot_cirurgia_urgencia' => 0,
                'percen_cirurgia_urgencia' => 0,

                'tot_cirurgia_rotina' => 0,
                'percen_cirurgia_rotina' => 0,

                'tot_cirurgia_emergencia' => 0,
                'percen_cirurgia_emergencia' => 0,

                'tot_madrugada' => 0,
                'percen_madruaga' => 0,

                'tot_manha' => 0,
                'percen_manha' => 0,

                'tot_tarde' => 0,
                'percen_tarde' => 0,

                'tot_noite' => 0,
                'percen_noite' => 0,

                'tot_madrugada' => 0,
                'percen_madrugada' => 0,

                'specialties' => $specialties_arr,

                'dia_da_semana' => [
                    0 => ['tot' => 0, 'percen' => 0],
                    1 => ['tot' => 0, 'percen' => 0],
                    2 => ['tot' => 0, 'percen' => 0],
                    3 => ['tot' => 0, 'percen' => 0],
                    4 => ['tot' => 0, 'percen' => 0],
                    5 => ['tot' => 0, 'percen' => 0],
                    6 => ['tot' => 0, 'percen' => 0],
                ],

                'taxa_anti_60m_antes_cirurgia' => 0,
                'tot_anti_60m_antes_cirurgia' => 0,

                'taxa_anti_60m_antes_cirurgia_porte_pequeno' => 0,
                'tot_anti_60m_antes_cirurgia_porte_pequeno' => 0,

                'taxa_anti_60m_antes_cirurgia_porte_medio' => 0,
                'tot_anti_60m_antes_cirurgia_porte_medio' => 0,

                'taxa_anti_60m_antes_cirurgia_porte_grande' => 0,
                'tot_anti_60m_antes_cirurgia_porte_grande' => 0,
            ];
        }

        foreach ($hospitalizations as $hospitalization)
        {
            foreach ($hospitalization['surgeries'] as $surgery)
            {
                $data['total_de_cirurgias']++;

                array_push($data['surgeries'], $surgery);
                array_push($data['ids'], $surgery['id']);
                $turno_anestesia = null;
                $dia_da_cirurgia = null;
                $turno_cirurgia = null;

                if ($surgery['inicio_da_cirurgia_data'])
                {
                    $dia_da_cirurgia = new \DateTime($surgery['inicio_da_cirurgia_data']->format('Y-m-d'));                   
                    $data['dia_da_semana'][$dia_da_cirurgia->format('w')]['tot_cirurgias']++;                        
                }

                switch ($surgery['porte_da_cirurgia'])
                {
                    case 'Pequeno':
                        $data['tot_porte_da_cirurgia_pequeno']++;
                        break;

                    case 'Médio':
                        $data['tot_porte_da_cirurgia_medio']++;
                        break;

                    case 'Grande':
                        $data['tot_porte_da_cirurgia_grande']++;
                        break;
                }

                if (stripos($surgery['nome_da_cirurgia'], 'ortése') &&
                    stripos($surgery['nome_da_cirurgia'], 'prótese'))
                {
                    $data['tot_cirurgia_ortese_protese']++;
                }



                if ($surgery['inicio_da_anestesia_hora'])
                {
                    $hora_anestesia = strtotime($surgery['inicio_da_anestesia_hora']->format('H:i:s'));
                    
                    if ($hora_anestesia >= strtotime('00:00:00') &&
                        $hora_anestesia <= strtotime('06:00:00'))
                    {
                        $turno_anestesia = 'madrugada';
                    }
                    else if ($hora_anestesia >= strtotime('06:00:00') &&
                        $hora_anestesia <= strtotime('11:59:00'))
                    {
                        $turno_anestesia = 'manha';
                    }
                    else if ($hora_anestesia >= strtotime('12:00:00') &&
                        $hora_anestesia <= strtotime('17:59:00'))
                    {
                        $turno_anestesia = 'tarde';
                    }
                    else if ($hora_anestesia >= strtotime('18:00:00') &&
                        $hora_anestesia <= strtotime('23:59:00'))
                    {
                        $turno_anestesia = 'noite';
                    }                    

                    if ($surgery['termino_da_anestesia_hora'] &&
                        $surgery['inicio_da_cirurgia_data'])                
                    {
                        $data_hora_inicio_anestesia = new \DateTime($surgery['inicio_da_cirurgia_data']->format('Y-m-d') . ' ' . $surgery['inicio_da_anestesia_hora']->format('H:i:s'));
                        $data_hora_termino_anestesia = new \DateTime($surgery['inicio_da_cirurgia_data']->format('Y-m-d') . ' ' . $surgery['termino_da_anestesia_hora']->format('H:i:s'));

                        $tempo_anestesia = $data_hora_inicio_anestesia->diff($data_hora_termino_anestesia);
                        $tempo_anestesia = $tempo_anestesia->h + ($tempo_anestesia->i / 60) + ($tempo_anestesia->days * 24);

                        array_push($data['arr_tempo_realizacao_anestesia'], $tempo_anestesia);

                        if ($turno_cirurgia)
                        {
                            array_push($data['turnos'][$turno_cirurgia]['arr_tempo_realizacao_anestesia'], $tempo_anestesia);
                        }
                    }
                }

                if ($surgery['inicio_da_cirurgia_hora'])
                {
                    $hora_cirurgia = strtotime($surgery['inicio_da_cirurgia_hora']->format('H:i:s'));
                    
                    if ($hora_cirurgia >= strtotime('00:00:00') &&
                        $hora_cirurgia <= strtotime('06:00:00'))
                    {
                        $turno_cirurgia = 'madrugada';
                    }
                    else if ($hora_cirurgia >= strtotime('06:00:00') &&
                        $hora_cirurgia <= strtotime('11:59:00'))
                    {
                        $turno_cirurgia = 'manha';
                    }
                    else if ($hora_cirurgia >= strtotime('12:00:00') &&
                        $hora_cirurgia <= strtotime('17:59:00'))
                    {
                        $turno_cirurgia = 'tarde';
                    }
                    else if ($hora_cirurgia >= strtotime('18:00:00') &&
                        $hora_cirurgia <= strtotime('23:59:00'))
                    {
                        $turno_cirurgia = 'noite';
                    }

                    if ($turno_cirurgia)
                    {
                        $data['turnos'][$turno_cirurgia]['tot_cirurgias'] ++;    
                    }
                }

                if ($surgery['inicio_da_cirurgia_hora'] && 
                    $surgery['inicio_da_cirurgia_data'] &&
                    $surgery['realizado_antibiotico_profilaxia_hora'])
                {
                    $inicio_da_cirurgia_hora = new \DateTime($surgery['inicio_da_cirurgia_data']->format('Y-m-d') . ' ' . $surgery['inicio_da_cirurgia_hora']->format('H:i:s'));
                    $realizado_antibiotico_profilaxia_hora = new \DateTime($surgery['inicio_da_cirurgia_data']->format('Y-m-d') . ' ' . $surgery['realizado_antibiotico_profilaxia_hora']->format('H:i:s'));
                    $tempo_antibiotico = $realizado_antibiotico_profilaxia_hora->diff($inicio_da_cirurgia_hora);
                    $tempo_antibiotico = ($tempo_antibiotico->h * 60) + $tempo_antibiotico->i;

                    if ($tempo_antibiotico <= 60)
                    {
                        $data['tot_anti_60m_antes_cirurgia'] ++;

                        if ($turno_anestesia)
                        {
                            $data['turnos'][$turno_cirurgia]['tot_anti_60m_antes_cirurgia'] ++;    
                        }

                        if ($dia_da_cirurgia)
                        {
                            $data['dia_da_semana'][$dia_da_cirurgia->format('w')]['tot_anti_60m_antes_cirurgia']++;
                        }

                        //calc anestesia por porte da cirurgia
                        switch ($surgery['porte_da_cirurgia'])
                        {
                            case 'Pequeno':
                                $data['tot_anti_60m_antes_cirurgia_porte_pequeno'] ++;    
                                break;

                            case 'Médio':
                                $data['tot_anti_60m_antes_cirurgia_porte_medio'] ++;    
                                break;

                            case 'Grande':
                                $data['tot_anti_60m_antes_cirurgia_porte_grande'] ++;    
                                break;
                        }
                    }
                    else if ($tempo_antibiotico <= 90)
                    {
                        $data['tot_anti_90m_antes_cirurgia'] ++;                            
                    }  
                }

                //calc antibiotico 60m antes da cirurgia
                foreach ($surgery['specialties'] as  $specialty)
                {
                    $specialty_id = $specialty['id'];
                    $data['specialties'][$specialty_id]['tot_cirurgias']++;

                    if ($surgery['inicio_da_cirurgia_hora'] && 
                    $surgery['inicio_da_cirurgia_data'] &&
                    $surgery['realizado_antibiotico_profilaxia_hora'])
                    {
                        $inicio_da_cirurgia_hora = new \DateTime($surgery['inicio_da_cirurgia_data']->format('Y-m-d') . ' ' . $surgery['inicio_da_cirurgia_hora']->format('H:i:s'));
                        $realizado_antibiotico_profilaxia_hora = new \DateTime($surgery['inicio_da_cirurgia_data']->format('Y-m-d') . ' ' . $surgery['realizado_antibiotico_profilaxia_hora']->format('H:i:s'));
                        $tempo_antibiotico = $realizado_antibiotico_profilaxia_hora->diff($inicio_da_cirurgia_hora);
                        $tempo_antibiotico = ($tempo_antibiotico->h * 60) + $tempo_antibiotico->i;

                        if ($tempo_antibiotico <= 60)
                        {
                            $data['specialties'][$specialty_id]['tot_anti_60m_antes_cirurgia'] ++;    
                        }
                    }

                    if ($surgery['inicio_da_anestesia_hora'] &&
                        $surgery['termino_da_anestesia_hora'] &&
                        $surgery['inicio_da_cirurgia_data'])                
                    {
                        $data_hora_inicio_anestesia = new \DateTime($surgery['inicio_da_cirurgia_data']->format('Y-m-d') . ' ' . $surgery['inicio_da_anestesia_hora']->format('H:i:s'));
                        $data_hora_termino_anestesia = new \DateTime($surgery['inicio_da_cirurgia_data']->format('Y-m-d') . ' ' . $surgery['termino_da_anestesia_hora']->format('H:i:s'));

                        $tempo_anestesia = $data_hora_inicio_anestesia->diff($data_hora_termino_anestesia);
                        $tempo_anestesia = $tempo_anestesia->h + ($tempo_anestesia->i / 60) + ($tempo_anestesia->days * 24);

                        array_push($data['specialties'][$specialty_id]['arr_tempo_realizacao_anestesia'], $tempo_anestesia);

                        if ($turno_cirurgia)
                        {
                            array_push($data['specialties'][$specialty_id]['turnos'][$turno_cirurgia]['arr_tempo_realizacao_anestesia'], $tempo_anestesia);                            
                        }
                    }
                }

                foreach ($surgery['type_of_anesthesias'] as $anesthesia)
                {
                    $anesthesia_id = $anesthesia['id'];
                    $data['anestesias'][$anesthesia_id]['total']++;
                    $data['anestesias'][$anesthesia_id]['tot_cirurgias']++;

                    //calc uso de anestesia por turno
                    if ($turno_anestesia)
                    {
                        $data['anestesias'][$anesthesia_id]['tot_' . $turno_anestesia]++;
                    }

                    if ($dia_da_cirurgia)
                    {
                        $data['anestesias'][$anesthesia_id]['dia_da_semana'][$dia_da_cirurgia->format('w')]['tot']++;
                    }

                    //calc anestesia por porte da cirurgia
                    switch ($surgery['porte_da_cirurgia'])
                    {
                        case 'Pequeno':
                            $data['anestesias'][$anesthesia_id]['tot_porte_da_cirurgia_pequeno']++;
                            break;

                        case 'Médio':
                            $data['anestesias'][$anesthesia_id]['tot_porte_da_cirurgia_medio']++;
                            break;

                        case 'Grande':
                            $data['anestesias'][$anesthesia_id]['tot_porte_da_cirurgia_grande']++;
                            break;
                    }

                    //calc anestesia por caracteristica da cirurgia
                    switch ($surgery['caracteristica_da_cirurgia'])
                    {
                        case 'Urgência':
                            $data['anestesias'][$anesthesia_id]['tot_cirurgia_urgencia']++;
                            break;

                        case 'Rotina':
                            $data['anestesias'][$anesthesia_id]['tot_cirurgia_rotina']++;
                            break;

                        case 'Emergência':
                            $data['anestesias'][$anesthesia_id]['tot_cirurgia_emergencia']++;
                            break;
                    }

                    //calc anestesia por especialidade da cirurgia
                    foreach ($surgery['specialties'] as $specialty)
                    {
                        $specialty_id = $specialty['id'];
                        $data['anestesias'][$anesthesia_id]['specialties'][$specialty_id]['tot']++;
                    }
                    
                    if ($surgery['inicio_da_cirurgia_hora'] && 
                        $surgery['inicio_da_cirurgia_data'] &&
                        $surgery['realizado_antibiotico_profilaxia_hora'])
                    {
                        $inicio_da_cirurgia_hora = new \DateTime($surgery['inicio_da_cirurgia_data']->format('Y-m-d') . ' ' . $surgery['inicio_da_cirurgia_hora']->format('H:i:s'));
                        $realizado_antibiotico_profilaxia_hora = new \DateTime($surgery['inicio_da_cirurgia_data']->format('Y-m-d') . ' ' . $surgery['realizado_antibiotico_profilaxia_hora']->format('H:i:s'));
                        $tempo_antibiotico = $realizado_antibiotico_profilaxia_hora->diff($inicio_da_cirurgia_hora);
                        $tempo_antibiotico = ($tempo_antibiotico->h * 60) + $tempo_antibiotico->i;

                        if ($tempo_antibiotico <= 60)
                        {
                            $data['anestesias'][$anesthesia_id]['tot_anti_60m_antes_cirurgia'] ++;    
                        }
                        else if ($tempo_antibiotico <= 90)
                        {
                            $data['anestesias'][$anesthesia_id]['tot_anti_90m_antes_cirurgia'] ++;    
                        }                     
                    }
                }


                if ($data['cirurgia_contaminada'])
                {
                    $data['total_de_cirurgias_contaminadas']++;
                }

                foreach ($surgery['specialties'] as $specialty)
                {
                    $specialty_id = $specialty['id'];
                    $data['main_specialties'][$specialty_id]['tot']++;
                }                
            }
        }
        
        $data['main_specialties'] = $this->sortArray(5, $data['main_specialties']);
        $data['main_specialties_ids'] = [];

        foreach ($data['main_specialties'] as $value) 
        {
            if (isset($value['id']))
            {
                array_push($data['main_specialties_ids'], $value['id']);    
            }            
        }

        foreach ($hospitalization['infections'] as $infection)
        {              
            if ( $infection['IRAS_description'] == 6)
            {
                $data['tot_infeccao_ortese_protese']++;
            }            
        }

      

        foreach ($data['anestesias'] as $key => $anestesia)
        {
            //calc a percen da anestesia por porte da cirurgia
            if ($anestesia['tot_porte_da_cirurgia_pequeno'] > 0)
            {
                $data['anestesias'][$key]['percen_porte_da_cirurgia_pequeno'] = ($anestesia['tot_porte_da_cirurgia_pequeno'] * 100) / $anestesia['total'];
                $data['anestesias'][$key]['percen_porte_da_cirurgia_pequeno'] = round($data['anestesias'][$key]['percen_porte_da_cirurgia_pequeno'], 2);
            }

            if ($anestesia['tot_porte_da_cirurgia_medio'] > 0)
            {
                $data['anestesias'][$key]['percen_porte_da_cirurgia_medio'] = ($anestesia['tot_porte_da_cirurgia_medio'] * 100) / $anestesia['total'];
                $data['anestesias'][$key]['percen_porte_da_cirurgia_medio'] = round($data['anestesias'][$key]['percen_porte_da_cirurgia_medio'], 2);
            }

            if ($anestesia['tot_porte_da_cirurgia_grande'] > 0)
            {
                $data['anestesias'][$key]['percen_porte_da_cirurgia_grande'] = ($anestesia['tot_porte_da_cirurgia_grande'] * 100) / $anestesia['total'];
                $data['anestesias'][$key]['percen_porte_da_cirurgia_grande'] = round($data['anestesias'][$key]['percen_porte_da_cirurgia_grande'], 2);
            }

            //calc a percen da anestesia por caracteristica da cirurgia
            if ($anestesia['tot_cirurgia_urgencia'] > 0)
            {
                $data['anestesias'][$key]['percen_cirurgia_urgencia'] = ($anestesia['tot_cirurgia_urgencia'] * 100) / $anestesia['total'];
                $data['anestesias'][$key]['percen_cirurgia_urgencia'] = round($data['anestesias'][$key]['percen_cirurgia_urgencia'], 2);
            }

            if ($anestesia['tot_cirurgia_rotina'] > 0)
            {
                $data['anestesias'][$key]['percen_cirurgia_rotina'] = ($anestesia['tot_cirurgia_rotina'] * 100) / $anestesia['total'];
                $data['anestesias'][$key]['percen_cirurgia_rotina'] = round($data['anestesias'][$key]['percen_cirurgia_rotina'], 2);
            }

            if ($anestesia['tot_cirurgia_emergencia'] > 0)
            {
                $data['anestesias'][$key]['percen_cirurgia_emergencia'] = ($anestesia['tot_cirurgia_emergencia'] * 100) / $anestesia['total'];
                $data['anestesias'][$key]['percen_cirurgia_emergencia'] = round($data['anestesias'][$key]['percen_cirurgia_emergencia'], 2);
            }

            //calc percen por turno
            foreach (['manha', 'tarde', 'noite', 'madrugada'] as $turno)
            {
                if ($anestesia['tot_' . $turno])
                {
                    $data['anestesias'][$key]['percen_' . $turno] = ($anestesia['tot_' . $turno] * 100) / $anestesia['total'];
                    $data['anestesias'][$key]['percen_' . $turno] = round($data['anestesias'][$key]['percen_' . $turno], 2);
                }
            }

            foreach ($anestesia['dia_da_semana'] as $key_dia => $dia)
            {
                if ($dia['tot'] > 0)
                {
                    $data['anestesias'][$key]['dia_da_semana'][$key_dia]['percen'] = ($dia['tot'] * 100) / $anestesia['total'];
                    $data['anestesias'][$key]['dia_da_semana'][$key_dia]['percen'] = round($data['anestesias'][$key]['dia_da_semana'][$key_dia]['percen'], 2);
                }
            }

            foreach ($anestesia['specialties'] as $specialty_key => $specialty) 
            {                
                if (!in_array($specialty_key, $data['main_specialties_ids']))
                {
                    if (isset($data['anestesias'][$key]['specialties']['outras']))
                    {
                        $data['anestesias'][$key]['specialties']['outras']['tot'] += $specialty['tot'];
                        $data['anestesias'][$key]['specialties']['outras']['percen'] += $specialty['percen'];                        
                    }
                    else
                    {
                        $data['anestesias'][$key]['specialties']['outras'] = [
                            'name' => 'Outras',
                            'tot' => $specialty['tot'],
                            'percen' => $specialty['percen'],
                        ];
                    }

                    unset($data['anestesias'][$key]['specialties'][$specialty_key]);
                }                
            }

            if ($anestesia['tot_anti_60m_antes_cirurgia'] > 0 )
            {
                $data['anestesias'][$key]['taxa_anti_60m_antes_cirurgia'] = ($anestesia['tot_anti_60m_antes_cirurgia'] / $anestesia['tot_cirurgias']) * 100; 
                $data['anestesias'][$key]['taxa_anti_60m_antes_cirurgia'] = round($data['anestesias'][$key]['taxa_anti_60m_antes_cirurgia'], 2);
            }
        }

        if ($data['tot_infeccao_ortese_protese'] > 0 && $data['tot_cirurgia_ortese_protese'] > 0)
        {
            $data['taxa_infeccao_ortese_protese'] = round( $data['tot_infeccao_ortese_protese'] / $data['tot_cirurgia_ortese_protese'], 2);
        }
        

        if ($data['total_de_cirurgias_contaminadas'] > 0)
        {
            $data['taxa_de_cirurgias_contaminadas'] = round($data['total_de_cirurgias_contaminadas'] / $data['total_de_cirurgias'], 2);
        }

        if ($data['tot_anti_90m_antes_cirurgia'] > 0 )
        {
            $data['taxa_anti_90m_antes_cirurgia'] = ($data['tot_anti_90m_antes_cirurgia'] / $data['total_de_cirurgias']) * 100; 
            $data['taxa_anti_90m_antes_cirurgia'] = round($data['taxa_anti_90m_antes_cirurgia'], 2);
        }

        if ($data['tot_anti_60m_antes_cirurgia'] > 0 )
        {
            $data['taxa_anti_60m_antes_cirurgia'] = ($data['tot_anti_60m_antes_cirurgia'] / $data['total_de_cirurgias']) * 100; 
            $data['taxa_anti_60m_antes_cirurgia'] = round($data['taxa_anti_60m_antes_cirurgia'], 2);
        }

        if ($data['tot_anti_60m_antes_cirurgia_porte_pequeno'] > 0 )
        {
            $data['taxa_anti_60m_antes_cirurgia_porte_pequeno'] = ($data['tot_anti_60m_antes_cirurgia_porte_pequeno'] / $data['tot_porte_da_cirurgia_pequeno']) * 100; 
            $data['taxa_anti_60m_antes_cirurgia_porte_pequeno'] = round($data['taxa_anti_60m_antes_cirurgia_porte_pequeno'], 2);
        }
        if ($data['tot_anti_60m_antes_cirurgia_porte_medio'] > 0 )
        {
            $data['taxa_anti_60m_antes_cirurgia_porte_medio'] = ($data['tot_anti_60m_antes_cirurgia_porte_medio'] / $data['tot_porte_da_cirurgia_medio']) * 100; 
            $data['taxa_anti_60m_antes_cirurgia_porte_medio'] = round($data['taxa_anti_60m_antes_cirurgia_porte_medio'], 2);
        }
        if ($data['tot_anti_60m_antes_cirurgia_porte_grande'] > 0 ) 
        {
            $data['taxa_anti_60m_antes_cirurgia_porte_grande'] = ($data['tot_anti_60m_antes_cirurgia_porte_grande'] / $data['tot_porte_da_cirurgia_grande']) * 100; 
            $data['taxa_anti_60m_antes_cirurgia_porte_grande'] = round($data['taxa_anti_60m_antes_cirurgia_porte_grande'], 2);
        }

        foreach ($data['specialties'] as $specialty_key => $specialty) 
        {
            if ($specialty['tot_anti_60m_antes_cirurgia'] > 0)
            {
                $data['specialties'][$specialty_key]['taxa_anti_60m_antes_cirurgia'] = ($specialty['tot_anti_60m_antes_cirurgia'] / $specialty['tot_cirurgias'] ) * 100;
                $data['specialties'][$specialty_key]['taxa_anti_60m_antes_cirurgia'] = round($data['specialties'][$specialty_key]['taxa_anti_60m_antes_cirurgia'], 2);
            }
            $data['specialties'][$specialty_key]['media_tempo_realizacao_anestesia'] = $calc_obj->calc_average($specialty['arr_tempo_realizacao_anestesia']);

            foreach ($specialty['turnos'] as $turno_key => $turno) 
            {
                $data['specialties'][$specialty_key]['turnos'][$turno_key]['media_tempo_realizacao_anestesia'] = $calc_obj->calc_average($turno['arr_tempo_realizacao_anestesia']);
            }
        }

        foreach ($data['turnos'] as $turno_key => $turno) 
        {
            if ($turno['tot_anti_60m_antes_cirurgia'] > 0)
            {
                $data['turnos'][$turno_key]['taxa_anti_60m_antes_cirurgia'] = ($turno['tot_anti_60m_antes_cirurgia'] / $turno['tot_cirurgias']) *100;
                $data['turnos'][$turno_key]['taxa_anti_60m_antes_cirurgia'] = round($data['turnos'][$turno_key]['taxa_anti_60m_antes_cirurgia'], 2);
            }

             $data['turnos'][$turno_key]['media_tempo_realizacao_anestesia'] = $calc_obj->calc_average($turno['arr_tempo_realizacao_anestesia']);
        }

        foreach ($data['dia_da_semana'] as $dia_key => $dia) 
        {
            if ($dia['tot_anti_60m_antes_cirurgia'] > 0)
            {
                $data['dia_da_semana'][$dia_key]['taxa_anti_60m_antes_cirurgia'] = ($dia['tot_anti_60m_antes_cirurgia'] / $dia['tot_cirurgias']) *100;
                $data['dia_da_semana'][$dia_key]['taxa_anti_60m_antes_cirurgia'] = round($data['dia_da_semana'][$dia_key]['taxa_anti_60m_antes_cirurgia'], 2);
            }
        }

        // if (count($data['arr_tempo_realizacao_anestesia']) > 0)
        // {
            $data['media_tempo_realizacao_anestesia'] = $calc_obj->calc_average($data['arr_tempo_realizacao_anestesia']);
       // }

        return $data;
    }    

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules)
    {
        $rules->add($rules->existsIn(['hospitalization_id'], 'Hospitalizations'));
        return $rules;
    }

    private function sortArray($max_itens = 0, array $array)
    {
        //ordena as principais especialdiades
        usort($array, function ($a, $b)
        {
            return $a['tot'] < $b['tot'];
        });

        //seleciona somente as 5 principais especialidades
        $count = 0;
        foreach ($array as $key => $value)
        {
            $count++;
            if ($count > $max_itens && $value['name'] !== 'Outras')
            {                
                unset($array[$key]);
                if (isset($array['outras']['tot']))
                {
                    $array['outras']['tot']  += $value['tot'];
                    $array['outras']['percen']  += $value['percen'];
                }
                else
                {   
                    $array['outras'] = [
                        'name' => 'Outras',
                        'tot' => $value['tot'],
                        'percen' => $value['percen'],
                    ];
                }
            }
        }    
        return $array;
    }

    public function diaSemana($data)
    {
        $diasemana = ['Domingo', 'Segunda', 'Terça', 'Quarta', 'Quinta', 'Sexta', 'Sabado'];

        // Aqui podemos usar a data atual ou qualquer outra data no formato Ano-mês-dia (2014-02-28)
        $data = date($data);

        // Varivel que recebe o dia da semana (0 = Domingo, 1 = Segunda ...)
        $diasemana_numero = date('w', strtotime($data));

        // Exibe o dia da semana com o Array
        return $diasemana[$diasemana_numero];
    }
}
